import {batchActions} from 'redux-batched-actions';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {TCountryKey, TSettlementKey, TStationKey} from 'types/PointKey';
import {IAviaGetSearchToCountryPageApiResponse} from 'server/api/AviaGatewayApi/types/IAviaGetSearchToCountryPageApiResponse';
import {IAviaSearchToCountryPageParams} from 'projects/avia/pages/AviaSearchToCountry/types/IAviaSearchToCountryPageParams';

import {
    CustomDispatch,
    CustomThunkAction,
    TGetState,
} from 'reducers/trains/customDispatch';
import * as actions from 'reducers/avia/searchToCountry/actions';
import {getAviaSearchFormActions} from 'reducers/avia/utils/fillSearchForm';
import {IAviaContext} from 'reducers/avia/data-types';
import {setAviaContext} from 'reducers/avia/context/actions';
import {getCountryRestrictionsActions} from 'reducers/avia/countryRestrictions/actions';

import {getOneWayParam} from 'projects/avia/lib/urls/getSearchUrl';
import {getSuggestByPointKey} from 'projects/avia/lib/getSuggestByPointKey';
import {isCompatibleRequestParams} from 'projects/avia/lib/searchToCountry/isCompatibleRequestParams';
import {unknownErrToString} from 'utilities/error';

import {aviaProvider} from 'serviceProvider/avia/aviaProvider';
import {aviaGeoProvider} from 'serviceProvider/avia/aviaGeoProvider';

interface IRequestSearchToCountryThunkAction
    extends IAviaSearchToCountryPageParams {
    req?: Request;
    res?: Response;
}

export default function requestSearchToCountryThunkAction({
    req,
    res,
    ...requestParams
}: IRequestSearchToCountryThunkAction): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const [toCountryInfo] = await Promise.all([
            getSearchToCountryPage(req, dispatch, getState, requestParams),
            prefillSearchFormAndContext(req, dispatch, requestParams),
        ]);

        if (toCountryInfo) {
            await getCountryRestrictions(
                req,
                requestParams.from,
                toCountryInfo.toCountry.key,
                dispatch,
            );
        }
    };
}

async function getSearchToCountryPage(
    req: Request | undefined,
    dispatch: CustomDispatch,
    getState: TGetState,
    requestParams: IAviaSearchToCountryPageParams,
): Promise<IAviaGetSearchToCountryPageApiResponse | undefined> {
    dispatch(actions.request({requestParams}));

    try {
        const aviaService = aviaProvider.provider(req?.container);
        const geoService = aviaGeoProvider.provider(req?.container);
        const [toCountryInfo, from] = await Promise.all([
            aviaService.getSearchToCountryPage(requestParams),
            geoService.getDataByPointKey(requestParams.from),
        ]);

        if (!from?.country) {
            throw new Error(
                'не удалось получить информацию о стране пункта отправления',
            );
        }

        const {
            avia: {
                searchToCountry: {requestParams: currentRequestParams},
            },
        } = getState();

        if (
            currentRequestParams &&
            !isCompatibleRequestParams(currentRequestParams, requestParams)
        ) {
            return;
        }

        dispatch(
            actions.requestSuccess({
                requestParams,
                response: toCountryInfo,
                fromPointCountryTitle: from.country.title,
            }),
        );

        return toCountryInfo;
    } catch (err) {
        dispatch(actions.requestFailure({error: unknownErrToString(err)}));

        throw err;
    }
}

async function getCountryRestrictions(
    req: Request | undefined,
    fromPointKey: TSettlementKey | TStationKey,
    toPointKey: TSettlementKey | TCountryKey,
    dispatch: CustomDispatch,
): Promise<void> {
    dispatch(getCountryRestrictionsActions.request());

    try {
        const countryRestrictionsResponse = await aviaProvider
            .provider(req?.container)
            .getExtendedCountryRestrictions({
                fromPointKey,
                toPointKey,
            });

        dispatch(
            getCountryRestrictionsActions.success(countryRestrictionsResponse),
        );
    } catch (e) {
        dispatch(getCountryRestrictionsActions.failure(e));
    }
}

async function prefillSearchFormAndContext(
    req: Request | undefined,
    dispatch: CustomDispatch,
    requestParams: IAviaSearchToCountryPageParams,
): Promise<void> {
    const geoService = aviaGeoProvider.provider(req?.container);
    const {
        from,
        to,
        passengers: {adults, children, infants},
        klass,
        dateForward,
        dateBackward,
    } = requestParams;

    const [fromSuggest, toSuggest] = await Promise.all([
        getSuggestByPointKey(geoService, from),
        getSuggestByPointKey(geoService, to),
    ]);

    const aviaContext: IAviaContext = {
        from: fromSuggest,
        fromId: from,
        fromName: fromSuggest?.title || '',
        to: toSuggest,
        toId: to,
        toName: toSuggest?.title || '',
        adult_seats: String(adults),
        children_seats: String(children),
        infant_seats: String(infants),
        klass,
        when: dateForward,
        return_date: dateBackward || '',
        oneway: getOneWayParam(dateBackward),
    };

    dispatch(
        batchActions([
            ...getAviaSearchFormActions(aviaContext),
            setAviaContext(aviaContext),
        ]),
    );
}
