import assign from 'lodash/fp/assign';

import {mergeReferences} from 'reducers/avia/utils/ticketDaemon/mergeReferences';
import {INormalizedTDAnswer} from 'reducers/avia/utils/ticketDaemon/normalizeTDAnswer';
import {cleanUpOutdatedVariants} from 'reducers/avia/utils/ticketDaemon/cleanUpOutdatedVariants';
import {INormalizedPreparedTDAnswer} from 'reducers/avia/utils/ticketDaemon/normalizedPreparedTDAnswer';

import {processNormalizedVariants} from 'selectors/avia/search/getDenormalizedVariants';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

export function mergeTDAnswer<S extends INormalizedPreparedTDAnswer>(
    state: S,
    payload: INormalizedTDAnswer,
    withCleanUp: boolean = true,
): S {
    const {cont, variants, reference, progress, partners} = payload;

    const resultVariants = processNormalizedVariants(variants, reference);

    const variantsMap: Record<string, IResultAviaVariant> = {};
    const variantGroups: Record<string, string[]> = {};

    resultVariants.forEach(variant => {
        const {key, tag} = variant;

        if (variantsMap[tag]) {
            if (
                variant.price.tariff.value < variantsMap[tag].price.tariff.value
            ) {
                variantsMap[tag] = variant;
            }
        } else {
            variantsMap[tag] = variant;
        }

        if (!variantGroups[key]) {
            variantGroups[key] = (state.variantGroups[key] || []).slice();
        }

        variantGroups[key].push(tag);
    });

    const nextVariants = assign(state.variants, variantsMap);
    const nextPartners = assign(state.partners, partners);
    const nextVariantGroups = assign(state.variantGroups, variantGroups);

    // Необходимо фильтровать каждый ответ бекенда, т.к. партнёр помеченный в начале
    // как outdated в конце опроса может прийти со статусом done, и мы не сможем
    // определить какие варианты нужно было выбросить
    if (withCleanUp) {
        cleanUpOutdatedVariants(nextPartners, nextVariants, nextVariantGroups);
    }

    return {
        ...state,
        cont: cont,
        searchIsCompleted: cont === null,
        variants: nextVariants,
        variantGroups: nextVariantGroups,
        reference: mergeReferences(state.reference, reference),
        progress,
        partners: nextPartners,
    };
}
