import {get} from 'lodash';

import {
    TFareFamilies,
    EFareFamilyTermCode,
    IFFBaggageTerm,
    IFFMilesTerm,
    IFFChangingCarriageTerm,
    IFFCarryOnTerm,
    IFFRefundableTerm,
    TFareFamilyKey,
    TTerm,
    TTariffGroupNameLang,
    IFFBaggageTermRule,
    IFFMilesTermRule,
    IFFRefundableTermRule,
    IFFChangingCarriageTermRule,
    IFFCarryOnTermRule,
    IFFDisclosureUrlTerm,
    IFFSeatSelectionCheckInTerm,
    IFFSeatSelectionCheckInTermRule,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';
import {EAviaClassType} from 'types/avia/EAviaClassType';

export type TNormalizedFareFamilies = Record<
    TFareFamilyKey,
    INormalizedFareFamily
>;

export interface INormalizedFareFamily {
    baseClass: EAviaClassType;
    tariffGroupName: string;
    terms: INormalizedTerms;
}

export interface INormalizedTerms {
    [EFareFamilyTermCode.baggage]?: IFFBaggageTermRule;
    [EFareFamilyTermCode.miles]?: IFFMilesTermRule;
    [EFareFamilyTermCode.refundable]?: IFFRefundableTermRule;
    [EFareFamilyTermCode.changingCarriage]?: IFFChangingCarriageTermRule;
    [EFareFamilyTermCode.carryOn]?: IFFCarryOnTermRule;
    [EFareFamilyTermCode.disclosureUrl]?: string;
    [EFareFamilyTermCode.seatSelectionCheckIn]?: IFFSeatSelectionCheckInTermRule;
}

export function normalizeFareFamilies(
    families: TFareFamilies,
    lang: TTariffGroupNameLang = 'ru',
): TNormalizedFareFamilies {
    return Object.keys(families).reduce((normalizedFamilies, familyName) => {
        const family = families[familyName];

        const terms = family.terms.reduce((normalizedTerms, term) => {
            if (checkBaggageTerm(term)) {
                normalizedTerms[EFareFamilyTermCode.baggage] = term.rule;
            }

            if (checkMilesTerm(term)) {
                normalizedTerms[EFareFamilyTermCode.miles] = term.rule;
            }

            if (checkRefundableTerm(term)) {
                normalizedTerms[EFareFamilyTermCode.refundable] = term.rule;
            }

            if (checkChangingCarriageTerm(term)) {
                normalizedTerms[EFareFamilyTermCode.changingCarriage] =
                    term.rule;
            }

            if (checkCarryOnTerm(term)) {
                normalizedTerms[EFareFamilyTermCode.carryOn] = term.rule;
            }

            if (checkDisclosureUrlTerm(term)) {
                // сейчас в данных всегда только ru
                normalizedTerms[EFareFamilyTermCode.disclosureUrl] = get(term, [
                    'rule',
                    'special_notes',
                    0,
                    'ru',
                ]);
            }

            if (checkSeatSelectionCheckIn(term)) {
                normalizedTerms[EFareFamilyTermCode.seatSelectionCheckIn] =
                    term.rule;
            }

            return normalizedTerms;
        }, {} as INormalizedTerms);

        normalizedFamilies[familyName] = {
            baseClass:
                family.base_class === 'BUSINESS'
                    ? EAviaClassType.BUSINESS
                    : EAviaClassType.ECONOMY,
            tariffGroupName: family.tariff_group_name[lang],
            terms,
        };

        return normalizedFamilies;
    }, {} as TNormalizedFareFamilies);
}

function checkBaggageTerm(term: TTerm): term is IFFBaggageTerm {
    return term.code === EFareFamilyTermCode.baggage;
}

function checkMilesTerm(term: TTerm): term is IFFMilesTerm {
    return term.code === EFareFamilyTermCode.miles;
}

function checkRefundableTerm(term: TTerm): term is IFFRefundableTerm {
    return term.code === EFareFamilyTermCode.refundable;
}

function checkChangingCarriageTerm(
    term: TTerm,
): term is IFFChangingCarriageTerm {
    return term.code === EFareFamilyTermCode.changingCarriage;
}

function checkCarryOnTerm(term: TTerm): term is IFFCarryOnTerm {
    return term.code === EFareFamilyTermCode.carryOn;
}

function checkDisclosureUrlTerm(term: TTerm): term is IFFDisclosureUrlTerm {
    return term.code === EFareFamilyTermCode.disclosureUrl;
}

function checkSeatSelectionCheckIn(
    term: TTerm,
): term is IFFSeatSelectionCheckInTerm {
    return term.code === EFareFamilyTermCode.seatSelectionCheckIn;
}
