import {keyBy} from 'lodash';

import {MINUTE} from 'utilities/dateUtils/constants';

import {
    IReference,
    IAviaCompany,
    ICompany,
    IAlliance,
    IBaggageTDTariff,
    ICompanyTariff,
    IFlight,
    ITDPartner,
    IRating,
    TFlightNumber,
    ISettlement,
    IStation,
    IFlightDate,
    TStatus,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {
    TNormalizedFareFamilies,
    normalizeFareFamilies,
} from './normalizeFareFamilies';

export interface INormalizedTDReference {
    aviaCompanies: Record<string, IAviaCompany>;
    companies: Record<string, ICompany>;
    alliances: Record<string, IAlliance>;
    baggageTariffs: Record<string, IBaggageTDTariff>;
    companyTariffs: Record<string, ICompanyTariff>;
    flights: Record<string, IFlightWithPrecomputedTime>;
    partners: Record<string, INormalizedPartner>;
    reviewsCount: Record<TFlightNumber, number>;
    ratings: Record<TFlightNumber, IRating>;
    settlements: Record<string, ISettlement>;
    stations: Record<string, IStation>;
    fareFamilies: TNormalizedFareFamilies;
}

export interface INormalizedPartner extends ITDPartner {
    status: TStatus;
}

export interface IFlightWithPrecomputedTime extends IFlight {
    arrivalTime: number;
    departureTime: number;
}

export function normalizeTDReference(
    reference: IReference,
    partnersStatuses: Record<string, TStatus>,
): INormalizedTDReference {
    const {
        alliances,
        aviaCompanies,
        baggageTariffs,
        companies,
        companyTariffs,
        flights,
        partners,
        ratings,
        reviewsCount,
        settlements,
        stations,
        fareFamilies,
    } = reference;

    return {
        aviaCompanies: keyBy(aviaCompanies, 'id'),
        companies: keyBy(companies, 'id'),
        alliances: keyBy(alliances, 'id'),
        baggageTariffs,
        companyTariffs: keyBy(companyTariffs, 'id'),
        flights: keyBy(precomputeTimeForFlights(flights), 'key'),
        partners: keyBy(withPartnerStatus(partners, partnersStatuses), 'code'),
        reviewsCount,
        ratings: keyBy(ratings, 'number'),
        settlements: keyBy(settlements, 'id'),
        stations: keyBy(stations, 'id'),
        fareFamilies: normalizeFareFamilies(fareFamilies),
    };
}

function withPartnerStatus(
    partners: ITDPartner[],
    partnersStatuses: Record<string, TStatus>,
) {
    return partners.map(p => ({...p, status: partnersStatuses[p.code]}));
}

function precomputeTimeForFlights(
    flights: IFlight[],
): IFlightWithPrecomputedTime[] {
    return flights.map(
        (flight): IFlightWithPrecomputedTime => ({
            ...flight,
            arrivalTime: flight.arrival
                ? getTimeFromFlightDate(flight.arrival)
                : 0,
            departureTime: flight.departure
                ? getTimeFromFlightDate(flight.departure)
                : 0,
        }),
    );
}

function getTimeFromFlightDate({local, offset}: IFlightDate): number {
    return new Date(`${local}.000Z`).getTime() - offset * MINUTE;
}
