import isEqual from 'lodash/isEqual';
import {batchActions} from 'redux-batched-actions';

import {IBusesCreateRideOfferBrowserProviderParams} from 'serviceProvider/buses/types/IBusesCreateRideOfferBrowserProviderParams';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    changeRequestParams,
    createRideOfferActions,
} from 'reducers/buses/book/rideOffer/actions';

import rideOfferSelector from 'selectors/buses/book/rideOfferSelector';

import {logError} from 'utilities/logger/logError';
import updateLocationWithQuery from 'utilities/updateLocationWithQuery/updateLocationWithQuery';
import {reachGoal} from 'utilities/metrika';

import {busesBrowserProvider} from 'serviceProvider/buses/busesBrowserProvider';

export default function createRideOfferAction(
    params: IBusesCreateRideOfferBrowserProviderParams,
): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const state = getState();

            const {requestParams: prevRequestParams} = rideOfferSelector(state);

            if (isEqual(prevRequestParams, params)) {
                return;
            }

            dispatch(createRideOfferActions.request(params));

            const rideOffer = await busesBrowserProvider.createRideOffer(
                params,
            );

            const actions = [];

            /**
             * На бэке может происходить подмена райда на аналогичный (если со старым что-то произошло)
             */
            const hasRideIdChanged =
                rideOffer.ride && rideOffer.ride.id !== params.rideId;

            if (hasRideIdChanged) {
                actions.push(
                    changeRequestParams({
                        ...params,
                        rideId: rideOffer.ride.id,
                    }),
                );
            }

            actions.push(createRideOfferActions.success(rideOffer));

            dispatch(batchActions(actions));

            if (hasRideIdChanged) {
                updateLocationWithQuery(
                    {rideId: rideOffer.ride.id},
                    undefined,
                    {
                        replace: true,
                    },
                );

                reachGoal(EBusesGoal.BOOK_PAGE_RIDE_CHANGED);
            }
        } catch (err) {
            dispatch(createRideOfferActions.failure());

            logError(
                {
                    message: '[YATRAVEL][BUSES] Создание оффера',
                },
                err,
            );

            throw err;
        }
    };
}
