import {v4 as uuid} from 'uuid';

import {IBusesBookFormValues} from 'types/buses/booking/IBusesBookFormValues';
import {IBusesBookQuery} from 'types/buses/booking/IBusesBookQuery';
import {ICountry} from 'types/common/ICountry';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import prepareContacts from 'reducers/buses/book/thunk/createOrRestoreOrderAction/utilities/prepareContacts';
import getTickets from 'reducers/buses/book/thunk/createOrRestoreOrderAction/utilities/getTickets';
import tryRestoreOrder from 'reducers/buses/book/thunk/createOrRestoreOrderAction/utilities/tryRestoreOrder';
import saveTravelerAndPassengers from 'reducers/buses/book/thunk/createOrRestoreOrderAction/utilities/saveTravelerAndPassengers/saveTravelerAndPassengers';

import rideOfferSelector from 'selectors/buses/book/rideOfferSelector';

import {logError} from 'utilities/logger/logError';
import browserHistory from 'utilities/browserHistory/browserHistory';
import {busesURLs} from 'projects/buses/utilities/urls';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import getParsedQueryPlaces from 'projects/buses/pages/booking/BookPage/utilities/getParsedQueryPlaces';
import {getTestContextTokens} from 'projects/buses/utilities/testContext/getTestContextToken';
import setCurrentStoredOrderInfo from 'projects/buses/applications/BookApp/utilities/currentStoredOrder/setCurrentStoredOrderInfo';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

export default function createOrRestoreOrderAction({
    params: {formData, citizenships, label, onError},
}: {
    params: {
        formData: IBusesBookFormValues;
        citizenships: ICountry[];
        label: string;
        onError(): void;
    };
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const {value: rideOffer} = rideOfferSelector(getState());
            const {passengers, contacts} = formData;

            const {
                from,
                to,
                rideId,
                when,
                places: rawPlaces,
                placesCount,
            }: Partial<IBusesBookQuery> = getQueryByBrowserHistory();

            if (
                !rideOffer?.bookParams ||
                !passengers ||
                !contacts ||
                !from ||
                !to ||
                !rideId ||
                !when ||
                !placesCount ||
                !browserHistory
            ) {
                throw new Error('Нет необходимых параметров');
            }

            const places = getParsedQueryPlaces(rawPlaces);
            const preparedContacts = prepareContacts(contacts);
            const tickets = getTickets(passengers, citizenships, places);

            const constantCreateOrderParams = {
                contactInfo: preparedContacts,
                busesServices: [
                    {
                        offerId: rideOffer.offerId,
                        tickets,
                        busTestContextToken: getTestContextTokens(),
                    },
                ],
                trainServices: [],
                hotelServices: [],
                orderHistory: [],
            };

            let orderId = await tryRestoreOrder(constantCreateOrderParams);

            if (!orderId) {
                const orderInfo = await genericOrderBrowserProvider.createOrder(
                    {
                        deduplicationKey: uuid(),
                        label,
                        ...constantCreateOrderParams,
                    },
                );

                orderId = orderInfo.id;

                setCurrentStoredOrderInfo({
                    id: orderId,
                    createParams: constantCreateOrderParams,
                });

                saveTravelerAndPassengers(preparedContacts, tickets);
            }

            browserHistory.push(
                busesURLs.getBookPaymentUrl({
                    orderId,
                    from,
                    to,
                    when,
                    rideId,
                    placesCount,
                    places: rawPlaces,
                }),
            );
        } catch (err) {
            onError();

            logError(
                {
                    message: '[YATRAVEL][BUSES] Создание заказа',
                },
                err,
            );
        }
    };
}
