import {SECOND} from 'utilities/dateUtils/constants';

import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import EOriginalPaymentErrorCode from 'types/common/EOriginalPaymentErrorCode';
import {
    EBusesOrderError,
    EOrderPollingStatus,
} from 'reducers/buses/book/order/types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setOrderAction,
    setOrderErrorAction,
    setOrderPollingStatusAction,
} from 'reducers/buses/book/order/actions';
import startPaymentAction from 'reducers/buses/book/thunk/startPaymentAction';

import orderSelector from 'selectors/buses/book/orderSelector';
import orderPollingStatusSelector from 'selectors/buses/book/orderPollingStatusSelector';

import {logError} from 'utilities/logger/logError';
import {delay} from 'utilities/async/delay';
import browserHistory from 'utilities/browserHistory/browserHistory';
import getHappyPageUrl from 'projects/buses/utilities/urls/getHappyPageUrl';
import removeCurrentStoredOrderInfo from 'projects/buses/applications/BookApp/utilities/currentStoredOrder/removeCurrentStoredOrderInfo';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

export default function startBookFlowAction({
    orderId,
}: {
    orderId: string;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const isQuerying = (): boolean =>
            orderPollingStatusSelector(getState()) ===
            EOrderPollingStatus.QUERYING;

        try {
            const stopPolling = (): void => {
                dispatch(
                    setOrderPollingStatusAction(EOrderPollingStatus.STOPPED),
                );
            };

            dispatch(setOrderPollingStatusAction(EOrderPollingStatus.QUERYING));

            let prevOrderState: EGenericOrderState | string | null = null;

            while (isQuerying()) {
                await delay(2 * SECOND);

                const {state: orderState} =
                    await genericOrderBrowserProvider.getOrderState({
                        orderId,
                    });

                if (prevOrderState === orderState) {
                    continue;
                }

                prevOrderState = orderState;

                const order = orderSelector(getState());

                if (
                    !order ||
                    orderState === EGenericOrderState.WAITING_PAYMENT ||
                    orderState === EGenericOrderState.PAYMENT_FAILED ||
                    orderState === EGenericOrderState.CANCELLED ||
                    orderState === EGenericOrderState.CONFIRMED
                ) {
                    const actualOrder =
                        await genericOrderBrowserProvider.getOrder({
                            orderId,
                        });

                    if (!isQuerying()) return;

                    dispatch(setOrderAction(actualOrder));
                }

                if (orderState === EGenericOrderState.RESERVED) {
                    await dispatch(startPaymentAction({orderId}));

                    continue;
                }

                if (orderState === EGenericOrderState.PAYMENT_FAILED) {
                    const updatedOrder = orderSelector(getState());

                    if (
                        updatedOrder?.payment?.errorInfo ===
                        EOriginalPaymentErrorCode.USER_CANCELLED
                    ) {
                        await dispatch(startPaymentAction({orderId}));

                        continue;
                    }

                    return stopPolling();
                }

                if (orderState === EGenericOrderState.CANCELLED) {
                    return stopPolling();
                }

                if (orderState === EGenericOrderState.CONFIRMED) {
                    browserHistory?.push(getHappyPageUrl({orderId}));
                    removeCurrentStoredOrderInfo();

                    return stopPolling();
                }
            }
        } catch (err) {
            if (!isQuerying()) return;

            dispatch(setOrderErrorAction(EBusesOrderError.COMMON));

            logError(
                {
                    message: '[YATRAVEL][BUSES] Флоу заказа',
                },
                err,
            );
        }
    };
}
