import isEqual from 'lodash/isEqual';

import {Request} from '@yandex-data-ui/core/lib/types';
import {isFilledContext} from 'types/buses/common/IBusesContext';
import {WHEN_SPECIAL_VALUE} from 'types/common/When';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    getContextActions,
    setContextValueAction,
} from 'reducers/buses/context/actions';

import requestContext, {
    IBusesRequestContextParams,
} from 'projects/buses/utilities/api/requestContext';
import getWhenMoment from 'utilities/dateUtils/when/getWhenMoment';
import {ROBOT} from 'utilities/dateUtils/formats';

/**
 * Чтобы присутствовали все поля, т.к.
 * isEqual({ a: 1, b: undefined }, { a: 1 }) === false
 */
function preparePrevParams(
    params: IBusesRequestContextParams | null,
): IBusesRequestContextParams | null {
    if (!params) {
        return params;
    }

    return {
        fromSlug: params.fromSlug,
        fromKey: params.fromKey,
        toSlug: params.toSlug,
        toKey: params.toKey,
    };
}

function getWhenAndOriginWhen(when: WHEN_SPECIAL_VALUE | string | undefined): {
    when: string | null;
    originWhen: WHEN_SPECIAL_VALUE | string | null;
} {
    return {
        when: when ? getWhenMoment(when).format(ROBOT) : null,
        originWhen: when || null,
    };
}

interface IRequestContextActionParams extends IBusesRequestContextParams {
    when: WHEN_SPECIAL_VALUE | string | undefined;
}

export default function requestContextAction({
    req,
    params,
}: {
    req?: Request;
    params: IRequestContextActionParams;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const {
                buses: {
                    context: {
                        value: prevValue,
                        requestParams: prevRequestParams,
                    },
                },
            } = getState();

            const {fromSlug, fromKey, toSlug, toKey, when} = params;

            const requestParams = {
                fromSlug,
                fromKey,
                toSlug,
                toKey,
            };

            if (isEqual(preparePrevParams(prevRequestParams), requestParams)) {
                if (prevValue && prevValue?.when !== when) {
                    dispatch(
                        setContextValueAction({
                            ...prevValue,
                            ...getWhenAndOriginWhen(when),
                        }),
                    );
                }

                return;
            }

            dispatch(getContextActions.request(requestParams));

            const apiContext = await requestContext({
                req,
                params: {
                    fromKey,
                    fromSlug,
                    toKey,
                    toSlug,
                },
            });

            const context = {
                ...apiContext,
                ...getWhenAndOriginWhen(when),
            };

            if (!isFilledContext(context)) {
                dispatch(getContextActions.failure());

                return;
            }

            dispatch(getContextActions.success(context));
        } catch (err) {
            dispatch(getContextActions.failure());
        }
    };
}
