import moment from 'moment-timezone';

import {CALENDAR_WIDTH_DAYS} from 'projects/buses/constants/calendar';

import {Request} from '@yandex-data-ui/core/lib/types';
import {IBusesFilledContext} from 'types/buses/common/IBusesContext';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {getSearchCalendarActions} from 'reducers/buses/search/calendar/actions';

import {getNow} from 'utilities/dateUtils';
import {ROBOT} from 'utilities/dateUtils/formats';
import requestCalendar from 'projects/buses/utilities/api/requestCalendar';
import getEarliestCalendarDateWithSegments from 'projects/buses/utilities/search/getEarliestCalendarDateWithSegments';
import requestSearchSegments from 'projects/buses/utilities/api/requestSearchSegments';

const CLOSEST_DAYS = [0, 1, 2];

export default function updateCalendarForClosestDaysThunkAction({
    req,
    calendarWidth = CALENDAR_WIDTH_DAYS,
    context,
}: {
    req?: Request;
    calendarWidth?: number;
    context: IBusesFilledContext;
}): CustomThunkAction<void> {
    return async dispatch => {
        const now = moment(getNow()).tz(context.from.timezone);

        const calendar = await requestCalendar({
            req,
            fromPointKey: context.from.key,
            toPointKey: context.to.key,
            maxDate: moment(now).add(calendarWidth, 'days').format(ROBOT),
        });

        const earliestDateWithSegments =
            getEarliestCalendarDateWithSegments(calendar);

        if (earliestDateWithSegments) {
            dispatch(getSearchCalendarActions.success(calendar));

            return;
        }

        const hasCacheForThreeClosestDays = CLOSEST_DAYS.some(day => {
            const momentForDay = moment(now).add(day, 'days');

            return calendar[momentForDay.format(ROBOT)];
        });

        if (hasCacheForThreeClosestDays) {
            dispatch(getSearchCalendarActions.success(calendar));

            return;
        }

        for (const day of CLOSEST_DAYS) {
            const date = moment(now).add(day, 'days').format(ROBOT);

            const contextForDay = {
                ...context,
                when: date,
                originWhen: date,
            };

            const segments = await requestSearchSegments({
                req,
                context: contextForDay,
            });

            if (segments.length) {
                dispatch(getSearchCalendarActions.success(calendar));

                return;
            }
        }

        dispatch(getSearchCalendarActions.success(calendar));
    };
}
