import {batchActions} from 'redux-batched-actions';

import {IPointInfo} from 'server/api/TransformSearchApi/types/IPointInfo';
import {
    EBusesSuggestType,
    IBusesSuggest,
} from 'types/buses/common/IBusesSuggest';
import {ITransformSearchContextRequestParams} from 'server/api/TransformSearchApi/types/ITransformSearchContextRequestParams';
import {IBusesSearchFormPointField} from 'projects/buses/components/SearchForm/types';
import {ITransformSearchContextResponse} from 'server/api/TransformSearchApi/types/ITransformSearchContextResponse';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setBusesSearchFormFromFieldAction,
    setBusesSearchFormStartDateFieldAction,
    setBusesSearchFormToFieldAction,
} from 'reducers/buses/searchForm/actions';

import {
    ESuggestSource,
    ISuggestValue,
} from 'components/SearchSuggest/SearchSuggest';

import {crossSearchProvider} from 'serviceProvider/common/crossSearch/crossSearchProvider';

const getBusesSuggestFromPointInfo = (
    pointInfo: IPointInfo,
): ISuggestValue<IBusesSuggest> => {
    return {
        inputValue: pointInfo.title,
        source: ESuggestSource.CROSS_SEARCH,
        selectedValue: {
            objId: pointInfo.pointKey ?? '',
            objType: EBusesSuggestType.SETTLEMENT,
            fullTitle: pointInfo.title,
            title: pointInfo.title,
            slug: pointInfo.slug ?? '',
            settlement: '',
            region: pointInfo.regionTitle ?? '',
            country: pointInfo.countryTitle,
            popularTitle: pointInfo.title,
            stationType: '',
            pointKey: pointInfo.pointKey ?? '',
            isUniqueTitle: true,
        },
    };
};

export const transformSearchContext = (
    params: Omit<ITransformSearchContextRequestParams, 'vertical'>,
): CustomThunkAction<Promise<ITransformSearchContextResponse>> => {
    return async (dispatch): Promise<ITransformSearchContextResponse> => {
        dispatch(
            batchActions([
                setBusesSearchFormFromFieldAction({
                    selectedValue: false,
                    inputValue: '',
                    source: ESuggestSource.CROSS_SEARCH,
                }),
                setBusesSearchFormToFieldAction({
                    selectedValue: false,
                    inputValue: '',
                    source: ESuggestSource.CROSS_SEARCH,
                }),
                setBusesSearchFormStartDateFieldAction(null),
            ]),
        );

        const crossSearchInfo = await crossSearchProvider
            .provider()
            .transformSearchContext({
                ...params,
                vertical: 'buses',
            });

        const {from, to, dateForward} = crossSearchInfo;

        const fromField: IBusesSearchFormPointField = from
            ? getBusesSuggestFromPointInfo(from)
            : {
                  selectedValue: false,
                  inputValue: '',
                  source: ESuggestSource.INPUT,
              };
        const toField: IBusesSearchFormPointField = to
            ? getBusesSuggestFromPointInfo(to)
            : {
                  selectedValue: false,
                  inputValue: '',
                  source: ESuggestSource.INPUT,
              };

        dispatch(
            batchActions([
                setBusesSearchFormFromFieldAction(fromField),
                setBusesSearchFormToFieldAction(toField),
                setBusesSearchFormStartDateFieldAction(dateForward ?? null),
            ]),
        );

        return crossSearchInfo;
    };
};
