import first from 'lodash/first';
import isEqual from 'lodash/isEqual';
import {batchActions} from 'redux-batched-actions';

import {ESearchFormFieldName} from 'components/SearchForm/types';
import {isNotUndefined} from 'types/utilities';
import {IBusesSearchFormPointField} from 'projects/buses/components/SearchForm/types';
import {Request} from '@yandex-data-ui/core/lib/types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setBusesSearchFormFromFieldAction,
    setBusesSearchFormToFieldAction,
} from 'reducers/buses/searchForm/actions';
import {
    getSearchSuggestsActions,
    IGetSearchSuggestsRequestAction,
} from 'reducers/buses/searchSuggests/actions';

import {getUserGeoLocation} from 'selectors/common/userInfoSelector';

import requestSearchSuggests from 'projects/buses/utilities/api/requestSearchSuggests';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

function requestAndSetSuggestsThunkAction({
    fieldType,
    requestParams,
    needToSetPointToSearchForm,
    req,
}: {
    fieldType: ESearchFormFieldName.FROM | ESearchFormFieldName.TO;
    requestParams: IGetSearchSuggestsRequestAction;
    needToSetPointToSearchForm: boolean;
    req?: Request;
}): CustomThunkAction<Promise<void>> {
    return async (dispatch, getState): Promise<void> => {
        try {
            dispatch(
                getSearchSuggestsActions.request({
                    fieldType,
                    requestParams,
                }),
            );

            const {items} = await requestSearchSuggests({
                ...requestParams,
                req,
            });

            const {requestParams: currentRequestParams} =
                getState().buses.searchSuggests[fieldType];

            /**
             * Если параллельно запустили еще один поиск, то устарелые данные нас не интересуют.
             */
            if (!isEqual(requestParams, currentRequestParams)) {
                return;
            }

            const firstSuggest = first(items);

            const setFormFieldAction =
                fieldType === ESearchFormFieldName.FROM
                    ? setBusesSearchFormFromFieldAction
                    : setBusesSearchFormToFieldAction;

            dispatch(
                batchActions(
                    [
                        getSearchSuggestsActions.success({
                            fieldType,
                            items: items,
                        }),
                        firstSuggest && needToSetPointToSearchForm
                            ? setFormFieldAction({
                                  inputValue: firstSuggest.title,
                                  source: ESuggestSource.SUGGESTS,
                                  selectedValue: firstSuggest,
                              })
                            : undefined,
                    ].filter(isNotUndefined),
                ),
            );
        } catch (err) {
            dispatch(getSearchSuggestsActions.failure({fieldType}));
        }
    };
}

export default function requestSearchSuggestsThunkAction({
    fromField,
    toField,
    needToSetByGeoPointIfPossible,
    req,
}: {
    fromField: IBusesSearchFormPointField;
    toField: IBusesSearchFormPointField;
    needToSetByGeoPointIfPossible: boolean;
    req?: Request;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const state = getState();

        const {geoId, geoName} = getUserGeoLocation(state);

        const {
            searchSuggests: {
                from: {requestParams: prevFromSuggestsRequestParams},
                to: {requestParams: prevToSuggestsRequestParams},
            },
        } = state.buses;

        const fromSuggestsRequestParams = {
            part: (
                fromField.inputValue ||
                (needToSetByGeoPointIfPossible ? geoName : '') ||
                ''
            ).toLowerCase(),
            clientCity: geoId ? String(geoId) : '',
            otherPoint: toField.selectedValue
                ? toField.selectedValue.objId
                : '',
        };

        if (
            !isEqual(prevFromSuggestsRequestParams, fromSuggestsRequestParams)
        ) {
            const requestFromSuggestsPromise = dispatch(
                requestAndSetSuggestsThunkAction({
                    fieldType: ESearchFormFieldName.FROM,
                    requestParams: fromSuggestsRequestParams,
                    needToSetPointToSearchForm: Boolean(
                        needToSetByGeoPointIfPossible &&
                            !fromField.selectedValue &&
                            geoName,
                    ),
                    req,
                }),
            );

            /**
             * Если саджест "Откуда" нужно выставить по геопозиции, то нужно дождаться, пока он загрузится, т.к.
             * от него зависят параметры для запроса саджеста "Куда", иначе можно загружать параллельно.
             */
            if (needToSetByGeoPointIfPossible) {
                await requestFromSuggestsPromise;
            }
        }

        const {from: updatedFromField} = getState().buses.searchForm;

        const toSuggestsRequestParams = {
            part: (toField.inputValue || '').toLowerCase(),
            clientCity: geoId ? String(geoId) : '',
            otherPoint: updatedFromField.selectedValue
                ? updatedFromField.selectedValue.objId
                : '',
        };

        if (!isEqual(prevToSuggestsRequestParams, toSuggestsRequestParams)) {
            await dispatch(
                requestAndSetSuggestsThunkAction({
                    fieldType: ESearchFormFieldName.TO,
                    requestParams: toSuggestsRequestParams,
                    needToSetPointToSearchForm: false,
                    req,
                }),
            );
        }
    };
}
