import {createReducer} from 'typesafe-actions';
import _uniq from 'lodash/uniq';

import {
    IRequiredOfferParams,
    IHotelOfferSearchProgress,
} from 'types/hotels/offer/IHotelOffer';
import IFavoriteCategory from 'projects/favorites/types/IFavoriteCategory';
import {
    IHotelWithOffers,
    THotelWithOffersByPermalink,
} from 'types/hotels/hotel/IHotelWithOffers';
import {IExtraVisitAndUserParams} from 'types/hotels/common/IExtraVisitAndUserParams';
import {PermalinkType} from 'types/hotels/hotel/IHotel';

import {getTotalNights} from 'projects/hotels/utilities/calculateTotalNights/calculateTotalNights';

import {
    FavoritesActionsTypes,
    changeFavoriteCategoryAction,
    pollingHotelsOffersActions,
    changeFavoriteHotelsOffsetAction,
    setCheckinDateAction,
    setCheckoutDateAction,
    setAdultsAction,
    setChildrenAgesAction,
    changeHotelsIsFavoriteActions,
    setShareTokenAction,
    getFavoritesInfoActions,
    getSharedFavoriteHotelsActions,
    setDefaultActiveHotel,
    setActiveHotel,
} from './actions';

export interface IFavoritesInfoState {
    offerSearchParams: IRequiredOfferParams | null;
    offerSearchProgress: IHotelOfferSearchProgress | null;
    nextPollingRequestDelayMs?: number;
    context: string | null;
    selectedCategoryId: string | null;
    categories: IFavoriteCategory[] | null;
    hotels: IHotelWithOffers[] | null;
    totalHotelCount: number | null;
    extraVisitAndUserParams: IExtraVisitAndUserParams | null;
    permalinksList: PermalinkType[] | null;
    hotelWithOffersByPermalink: THotelWithOffersByPermalink | null;
    isLoading: boolean;
    isError: boolean;
    lastSearchTimestamp: number | null;
    noMoreFavoritesModalIsOpen?: boolean;
    /** Категория этого списка (нужна для заголовка на странице) */
    category?: IFavoriteCategory;
    offset: number;
    activeHotelPermalink?: PermalinkType;
    defaultActiveHotelPermalink?: PermalinkType;
    shareToken: string;
    nights?: number;
}

const initialState: IFavoritesInfoState = {
    offerSearchParams: null,
    offerSearchProgress: null,
    context: null,
    selectedCategoryId: null,
    categories: null,
    hotels: null,
    totalHotelCount: null,
    extraVisitAndUserParams: null,
    permalinksList: null,
    hotelWithOffersByPermalink: null,
    isLoading: true,
    isError: false,
    lastSearchTimestamp: null,
    offset: 0,
    shareToken: '',
};

export default createReducer<IFavoritesInfoState, FavoritesActionsTypes>(
    initialState,
)
    .handleAction(getFavoritesInfoActions.request, state => ({
        ...state,
        isLoading: true,
    }))
    .handleAction(
        getFavoritesInfoActions.success,
        (state, {payload: {data, withInfiniteScroll}}) => {
            const {
                permalinksList: permalinksFromState,
                hotelWithOffersByPermalink: hotelWithOffersFromState,
            } = state;

            const newPermalinks = data.hotels.map(
                hotel => hotel.hotel.permalink,
            );
            const newHotelWithOffersByPermalink = data.hotels.reduce(
                (result, hotelWithOffers) => ({
                    ...result,
                    [hotelWithOffers.hotel.permalink]: hotelWithOffers,
                }),
                {},
            );

            const permalinksList = withInfiniteScroll
                ? _uniq([...(permalinksFromState || []), ...newPermalinks])
                : newPermalinks;
            const hotelWithOffersByPermalink = withInfiniteScroll
                ? {
                      ...hotelWithOffersFromState,
                      ...newHotelWithOffersByPermalink,
                  }
                : newHotelWithOffersByPermalink;

            return {
                ...state,
                ...data,
                selectedCategoryId: state.selectedCategoryId,
                permalinksList,
                hotelWithOffersByPermalink,
                isLoading: false,
                nights: getTotalNights(
                    data.offerSearchParams.checkinDate,
                    data.offerSearchParams.checkoutDate,
                ),
            };
        },
    )
    .handleAction(changeFavoriteCategoryAction, (state, {payload}) => ({
        ...state,
        selectedCategoryId: payload,
        offset: 0,
    }))
    .handleAction(pollingHotelsOffersActions.request, state => ({
        ...state,
        isLoading: true,
    }))
    .handleAction(
        pollingHotelsOffersActions.success,
        (state, {payload}): IFavoritesInfoState => {
            const {
                context,
                category,
                offers,
                offerSearchParams,
                offerSearchProgress,
            } = payload;

            const updatedHotelWithOffersByPermalink: Record<
                PermalinkType,
                IHotelWithOffers
            > = {};

            if (state.hotelWithOffersByPermalink) {
                Object.entries(state.hotelWithOffersByPermalink).forEach(
                    ([permalink, hotelInfo]) => {
                        updatedHotelWithOffersByPermalink[permalink] = {
                            ...hotelInfo,
                            ...(offers[permalink] || {}),
                        };
                    },
                );
            }

            return {
                ...state,
                isLoading: false,
                isError: false,
                lastSearchTimestamp: Date.now(),
                context,
                category,
                hotelWithOffersByPermalink: {
                    ...updatedHotelWithOffersByPermalink,
                },
                offerSearchParams,
                offerSearchProgress,
                nights: getTotalNights(
                    payload.offerSearchParams.checkinDate,
                    payload.offerSearchParams.checkoutDate,
                ),
            };
        },
    )
    .handleAction(pollingHotelsOffersActions.failure, state => ({
        ...state,
        isError: true,
        isLoading: false,
    }))
    .handleAction(changeFavoriteHotelsOffsetAction, (state, {payload}) => ({
        ...state,
        offset: payload,
    }))
    .handleAction(setCheckinDateAction, (state, {payload}) => {
        const {offerSearchParams} = state;

        if (offerSearchParams) {
            return {
                ...state,
                offerSearchParams: {
                    ...offerSearchParams,
                    checkinDate: payload,
                },
            };
        }

        return state;
    })
    .handleAction(setCheckoutDateAction, (state, {payload}) => {
        const {offerSearchParams} = state;

        if (offerSearchParams) {
            return {
                ...state,
                offerSearchParams: {
                    ...offerSearchParams,
                    checkoutDate: payload,
                },
            };
        }

        return state;
    })
    .handleAction(setAdultsAction, (state, {payload}) => {
        const {offerSearchParams} = state;

        if (offerSearchParams) {
            return {
                ...state,
                offerSearchParams: {
                    ...offerSearchParams,
                    adults: payload,
                },
            };
        }

        return state;
    })
    .handleAction(setChildrenAgesAction, (state, {payload}) => {
        const {offerSearchParams} = state;

        if (offerSearchParams) {
            return {
                ...state,
                offerSearchParams: {
                    ...offerSearchParams,
                    childrenAges: payload,
                },
            };
        }

        return state;
    })
    .handleAction(changeHotelsIsFavoriteActions.success, (state, {payload}) => {
        const {hotelWithOffersByPermalink} = state;

        if (hotelWithOffersByPermalink) {
            const selectedHoteIsFavorite =
                hotelWithOffersByPermalink[payload].hotel.isFavorite;

            const selectedHotelByPermalink = {
                ...hotelWithOffersByPermalink[payload].hotel,
                isFavorite: !selectedHoteIsFavorite,
            };

            return {
                ...state,
                hotelWithOffersByPermalink: {
                    ...hotelWithOffersByPermalink,
                    [payload]: {
                        ...hotelWithOffersByPermalink[payload],
                        hotel: selectedHotelByPermalink,
                    },
                },
                isLoading: false,
            };
        }

        return state;
    })
    .handleAction(setShareTokenAction, (state, {payload}) => ({
        ...state,
        shareToken: payload,
    }))
    .handleAction(getSharedFavoriteHotelsActions.request, state => ({
        ...state,
        isLoading: true,
    }))
    .handleAction(
        getSharedFavoriteHotelsActions.success,
        (state, {payload}) => ({
            ...state,
            ...payload,
            permalinksList: payload.hotels.map(hotel => hotel.hotel.permalink),
            hotelWithOffersByPermalink: payload.hotels.reduce(
                (result, hotel) => ({
                    ...result,
                    [hotel.hotel.permalink]: hotel,
                }),
                {},
            ),
            isLoading: false,
        }),
    )
    .handleAction(getSharedFavoriteHotelsActions.failure, state => ({
        ...state,
        isLoading: false,
        isError: true,
    }))
    .handleAction(setDefaultActiveHotel, (state, {payload}) => ({
        ...state,
        defaultActiveHotelPermalink: payload ? payload.permalink : undefined,
    }))
    .handleAction(setActiveHotel, (state, {payload}) => ({
        ...state,
        activeHotelPermalink: payload ? payload.permalink : undefined,
    }));
