import {createReducer} from 'typesafe-actions';

import {EOfferStatus} from 'projects/hotels/constants/EOfferStatus';

import {
    IBookOffer,
    TBookOfferError,
    IBookOfferInfoError,
    IBookOfferNotAvaliableError,
    IBookOfferNotFoundError,
    IParsedTokenInfo,
} from './types';

import {
    TOfferInfoByTokenActionTypes,
    fetchOfferInfoByTokenAction,
    updateOfferInfoAction,
    resetOfferInfoAction,
} from './actions';
import {stopOrderInfoPollingAction} from '../orderInfo/actions';

export interface IOfferInfoByTokenReducer {
    status: string;
    isError?: boolean;
    isSuccess: boolean;
    isLoading: boolean;
    offerInfo?: IBookOffer;
    error?: IBookOfferInfoError;
    tokenInfo?: IParsedTokenInfo;
}

const initialState: IOfferInfoByTokenReducer = {
    status: '',
    isError: false,
    isSuccess: false,
    isLoading: false,
};

function isNotAvaliableError(
    value: TBookOfferError,
): value is IBookOfferNotAvaliableError {
    return value.code === EOfferStatus.OFFER_IS_NOT_AVAILABLE;
}

function isNotFoundError(
    value: TBookOfferError,
): value is IBookOfferNotFoundError {
    return value.code === EOfferStatus.OFFER_NOT_FOUND_ERROR;
}

export default createReducer<
    IOfferInfoByTokenReducer,
    TOfferInfoByTokenActionTypes
>(initialState)
    .handleAction(fetchOfferInfoByTokenAction.request, () => ({
        ...initialState,
        isLoading: true,
    }))
    .handleAction(fetchOfferInfoByTokenAction.success, (state, {payload}) => {
        const {data} = payload;

        if (data) {
            /* Success offer */

            return {
                ...state,
                isSuccess: true,
                isLoading: false,
                status: EOfferStatus.OFFER_FETCHED,
                offerInfo: data,
            };
        }

        return {
            ...initialState,
            status: EOfferStatus.UNKNOWN_OFFER_ERROR,
            isError: true,
        };
    })
    .handleAction(fetchOfferInfoByTokenAction.failure, (_, {payload}) => {
        const {error} = payload;
        const {code: errorCode} = error || {};

        if (isNotAvaliableError(error) && error.offerInfo) {
            /* Fetched expired offer */

            return {
                status: EOfferStatus.OFFER_IS_NOT_AVAILABLE,
                isLoading: false,
                isSuccess: true,
                offerInfo: error.offerInfo,
            };
        }

        if (isNotFoundError(error) && error.tokenInfo) {
            /* Offer not found in cache */

            return {
                ...initialState,
                status: EOfferStatus.OFFER_NOT_FOUND_ERROR,
                isLoading: false,
                tokenInfo: error.tokenInfo,
            };
        }

        /* Fetched offer with errorCode */
        return {
            ...initialState,
            status: errorCode || EOfferStatus.UNKNOWN_OFFER_ERROR,
            isError: true,
        };
    })
    .handleAction(updateOfferInfoAction, (state, {payload}) => {
        const {data} = payload;

        return {
            ...state,
            offerInfo: data,
        };
    })
    .handleAction(stopOrderInfoPollingAction, state => ({
        ...state,
        isLoading: false,
    }))
    .handleAction(resetOfferInfoAction, () => initialState);
