import {createReducer} from 'typesafe-actions';

/* Types */
import {IHotelInfo} from './types';

import getUpdatedTextReviews from '../reviews/list/utilities/getUpdatedTextReviews';

import {
    HotelInfoActionsType,
    getHotelInfoActions,
    setHotelInfoReviewReactionAction,
    getHotelInfoOffersActions,
    setHotelInfoSearchParamsAction,
    getSimilarHotelsActions,
    setHotelInfoTotalReviewsAction,
    resetHotelInfoToInitialAction,
    changeHotelsIsFavoriteOnHotelPageActions,
    changeIsOpenForNoMoreFavoritesModalAction,
} from './actions';

export interface IHotelInfoReducer {
    isLoading: boolean;
    isLoadingOffers: boolean;
    isSuccess: boolean;
    isError: boolean;
    data?: IHotelInfo;
    noMoreFavoritesModalIsVisible?: boolean;
    status?: number;
}

/* Initial State */
const initialState: IHotelInfoReducer = {
    isLoading: true,
    isLoadingOffers: false,
    isSuccess: false,
    isError: false,
    noMoreFavoritesModalIsVisible: false,
};

/* Reducer */
export default createReducer<IHotelInfoReducer, HotelInfoActionsType>(
    initialState,
)
    .handleAction(getHotelInfoActions.request, () => ({
        isLoading: true,
        isLoadingOffers: false,
        isSuccess: false,
        isError: false,
    }))
    .handleAction(getHotelInfoActions.success, (state, {payload}) => {
        return {
            ...state,
            data: payload,
            isLoading: false,
            isSuccess: true,
            isError: false,
        };
    })
    .handleAction(getHotelInfoActions.failure, (state, {payload}) => {
        return {
            ...state,
            isLoading: false,
            isSuccess: false,
            isError: true,
            status: payload,
        };
    })
    .handleAction(setHotelInfoReviewReactionAction, (state, {payload}) => {
        const {data} = state;
        const {reviewId} = payload;

        if (data) {
            const {reviewsInfo} = data;
            const foundedReview = reviewsInfo.textReviews.find(
                review => review.id === reviewId,
            );

            if (foundedReview) {
                return {
                    ...state,
                    data: {
                        ...data,
                        reviewsInfo: {
                            ...reviewsInfo,
                            textReviews: getUpdatedTextReviews(
                                reviewsInfo.textReviews,
                                payload,
                            ),
                        },
                    },
                };
            }
        }

        return state;
    })
    .handleAction(getHotelInfoOffersActions.request, state => ({
        ...state,
        isLoadingOffers: true,
    }))
    .handleAction(getHotelInfoOffersActions.success, (state, {payload}) => {
        const {data} = state;

        if (data) {
            return {
                ...state,
                data: {
                    ...data,
                    offersInfo: payload,
                },
                isLoadingOffers: false,
            };
        }

        return state;
    })
    .handleAction(setHotelInfoSearchParamsAction, (state, {payload}) => {
        const {data} = state;

        if (data) {
            return {
                ...state,
                data: {
                    ...data,
                    searchParams: payload,
                },
            };
        }

        return state;
    })
    .handleAction(getSimilarHotelsActions.success, (state, {payload}) => {
        const {data} = state;

        if (data) {
            return {
                ...state,
                data: {
                    ...data,
                    similarHotelsInfo: payload,
                },
            };
        }

        return state;
    })
    .handleAction(setHotelInfoTotalReviewsAction, (state, {payload}) => {
        const {data} = state;
        const {totalTextReviewCount} = payload;

        if (data) {
            const {reviewsInfo, hotel} = data;

            if (reviewsInfo && hotel) {
                return {
                    ...state,
                    data: {
                        ...data,
                        hotel: {
                            ...hotel,
                            totalTextReviewCount,
                        },
                        reviewsInfo: {
                            ...reviewsInfo,
                            totalTextReviewCount,
                        },
                    },
                };
            }
        }

        return state;
    })
    .handleAction(
        resetHotelInfoToInitialAction,
        (): IHotelInfoReducer => initialState,
    )
    .handleAction(changeHotelsIsFavoriteOnHotelPageActions.success, state => {
        const {data} = state;

        if (data) {
            const {reviewsInfo, hotel} = data;

            if (reviewsInfo && hotel) {
                return {
                    ...state,
                    data: {
                        ...data,
                        hotel: {
                            ...hotel,
                            isFavorite: !hotel.isFavorite,
                        },
                    },
                };
            }
        }

        return state;
    })
    .handleAction(
        changeIsOpenForNoMoreFavoritesModalAction,
        (state, {payload}) => ({
            ...state,
            noMoreFavoritesModalIsVisible: payload,
        }),
    );
