import {
    MAIN_TAB_IMAGES_LIMIT,
    MAIN_TAB_SIMILAR_HOTELS_LIMIT,
} from 'projects/hotels/constants/hotelPage';
import {
    DEFAULT_BOT_TEXT_REVIEW_QUANTITATIVE,
    DEFAULT_TEXT_REVIEW_QUANTITATIVE,
} from 'projects/hotels/constants/reviews';

import {Request} from '@yandex-data-ui/core/lib/types';
import {EAddToFavoritesError} from 'server/api/HotelsFavoritesAPI/types/EAddToFavoritesError';
import IAddFavoriteHotelRequestParams from 'server/api/HotelsFavoritesAPI/types/IAddFavoriteHotelRequestParams';
import {IRemoveFavoriteHotelByPermalinkRequestParams} from 'server/api/HotelsFavoritesAPI/types/IRemoveFavoriteHotelsRequestParams';

import {setSnackbarInfo} from 'reducers/common/snackbar/actions';
import {
    getHotelReviewsActions,
    resetHotelReviewsAction,
} from 'reducers/hotels/hotelPage/reviews/list/actions';
import {CustomThunkAction} from 'reducers/trains/customDispatch';

import environmentConfigSelector from 'selectors/common/environmentConfigSelector';

import {isUnknownAxiosError} from 'utilities/error';
import {getAttributionParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';
import {getHotelIdentifier} from 'projects/hotels/utilities/getHotelIdentifier/getHotelIdentifier';
import {
    getHotelOfferToken,
    getOfferAttributionParamsByLocation,
} from 'projects/hotels/utilities/getHotelPageParams/getOfferAttributionParamsByLocation';
import {getHotelReviewsForBots} from 'projects/hotels/utilities/reviews/getHotelReviewsForBots';
import {
    validateSearchHotelsAdultsParam,
    validateSearchHotelsParams,
} from 'projects/hotels/utilities/validateSearchHotelsParams/validateSearchHotelsParams';

import {hotelSearchService, hotelsFavoritesService} from 'serviceProvider';

import {
    changeHotelsIsFavoriteOnHotelPageActions,
    changeIsOpenForNoMoreFavoritesModalAction,
    getHotelInfoActions,
} from './actions';

export const fetchHotelInfo = (
    req: Request,
): CustomThunkAction<Promise<void>> => {
    return async (dispatch, getState) => {
        try {
            const [, regionSlug, hotelSlug] = req.path
                ? req.path.split('/').filter(Boolean)
                : [];
            const queryParams = req.query;
            const reduxState = getState();
            const {tld} = environmentConfigSelector(reduxState);
            const reviewsParams = req.isBot
                ? DEFAULT_BOT_TEXT_REVIEW_QUANTITATIVE
                : DEFAULT_TEXT_REVIEW_QUANTITATIVE;

            const {
                hotelPermalink,
                adults,
                childrenAges,
                checkinDate,
                checkoutDate,
                searchPagePollingId,
                useDefaultSearchParams,
            } = validateSearchHotelsParams(queryParams);

            const optionsProps = {
                full: true,
            };

            const params = {
                adults: validateSearchHotelsAdultsParam(
                    adults,
                    checkinDate,
                    checkoutDate,
                ),
                childrenAges,
                checkinDate,
                checkoutDate,
                searchPagePollingId,
                useDefaultSearchParams,
                domain: tld,
                similarHotelLimit: MAIN_TAB_SIMILAR_HOTELS_LIMIT,
                imageLimit: MAIN_TAB_IMAGES_LIMIT,
                ...reviewsParams,
                ...getHotelIdentifier({hotelPermalink, regionSlug, hotelSlug}),
                ...getHotelOfferToken(queryParams),
                ...getOfferAttributionParamsByLocation(queryParams),
                ...optionsProps,
            };

            const hotelInfoResponse = await hotelSearchService
                .provider(req && req.container)
                .getHotelInfo(params);

            dispatch(getHotelInfoActions.success(hotelInfoResponse));
            dispatch(resetHotelReviewsAction());
            dispatch(
                getHotelReviewsActions.success(
                    req.isBot
                        ? getHotelReviewsForBots(hotelInfoResponse.reviewsInfo)
                        : hotelInfoResponse.reviewsInfo,
                ),
            );
        } catch (error) {
            const status = isUnknownAxiosError(error)
                ? error.response?.status
                : undefined;

            dispatch(getHotelInfoActions.failure(status));
        }
    };
};

export interface IAddFavoriteHotelParams {
    requestParams: IAddFavoriteHotelRequestParams;
    message?: React.ReactElement | string;
}

export const addFavoriteHotel = ({
    requestParams: {permalink},
    message,
}: IAddFavoriteHotelParams): CustomThunkAction<Promise<void>> => {
    return async dispatch => {
        try {
            const fillRequestParams: IAddFavoriteHotelRequestParams = {
                permalink,
                ...getAttributionParams(),
            };

            const result = await hotelsFavoritesService
                .provider()
                .addFavoriteHotel(fillRequestParams);

            if (
                'error' in result &&
                result.error &&
                result.error === EAddToFavoritesError.HOTEL_LIMIT_EXCEEDED
            ) {
                dispatch(changeIsOpenForNoMoreFavoritesModalAction(true));

                return;
            } else if ('error' in result && !result.error) {
                dispatch(changeHotelsIsFavoriteOnHotelPageActions.success());

                if (message) {
                    dispatch(
                        setSnackbarInfo({
                            lastAction: `${permalink}-add`,
                            hasCancellation: false,
                            message,
                            page: 'hotel',
                        }),
                    );
                }
            }
        } catch (error) {
            dispatch(changeHotelsIsFavoriteOnHotelPageActions.failure());
        }
    };
};

export const removeFavoriteHotel = ({
    permalink,
}: IRemoveFavoriteHotelByPermalinkRequestParams): CustomThunkAction<
    Promise<void>
> => {
    return async dispatch => {
        try {
            const requestParams = {
                permalink,
                ...getAttributionParams(),
            };

            await hotelsFavoritesService
                .provider()
                .removeFavoriteHotels(requestParams);

            dispatch(changeHotelsIsFavoriteOnHotelPageActions.success());
        } catch (error) {
            dispatch(changeHotelsIsFavoriteOnHotelPageActions.failure());
        }
    };
};
