import {
    TActiveFilterAtomsType,
    TFilterAtomType,
    IChangeFilterGroupPayload,
    IChangeFiltersTargetType,
    IChangePriceFilterPayload,
    ICurrentGeoIdFilterPayloadWithTarget,
    IFullFilters,
    IPriceFilter,
    IRangePriceFilter,
} from 'types/hotels/search/IFiltersInfo';
import {ISearchHotelsInfo} from 'server/api/HotelsSearchAPI/types/ISearchHotels';
import {IResetFilterAction} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {IHotelsFiltersReducer} from 'reducers/hotels/searchPage/filters/reducer';

export const fillPriceFilter = ({
    priceFilter,
    filterPriceFrom,
    filterPriceTo,
}: {
    priceFilter: IPriceFilter;
    filterPriceFrom?: number;
    filterPriceTo?: number;
}): IRangePriceFilter | undefined => {
    if (priceFilter) {
        return {
            minValue: filterPriceFrom || priceFilter.minPriceEstimate,
            maxValue: filterPriceTo || priceFilter.maxPriceEstimate,
            minPriceEstimate: priceFilter.minPriceEstimate,
            maxPriceEstimate: priceFilter.maxPriceEstimate,
            currency: priceFilter.currency,
        };
    }
};

export const convertFilterAtoms = (
    filterAtoms?: TFilterAtomType[],
): TActiveFilterAtomsType => {
    if (filterAtoms) {
        return filterAtoms.reduce(
            (activeFilterAtoms: TActiveFilterAtomsType, filterAtom) => {
                activeFilterAtoms[filterAtom] = true;

                return activeFilterAtoms;
            },
            {},
        );
    }

    return {};
};

export const fillFiltersBySearchInfo = (
    searchInfo: ISearchHotelsInfo,
): {
    permanentFilters: IFullFilters;
    currentFilters: IFullFilters;
} => {
    const {
        filterInfo: {
            params: {filterPriceFrom, filterPriceTo, filterAtoms},
            priceFilter,
            quickFilters,
            detailedFilters,
            detailedFiltersBatches,
            geoIdFilter,
        },
        searchControlInfo,
        foundHotelCount,
    } = searchInfo;
    const {quickControls = quickFilters} = searchControlInfo || {};

    const filters: IFullFilters = {
        priceFilter: fillPriceFilter({
            priceFilter,
            filterPriceFrom,
            filterPriceTo,
        }),
        activeFilterAtoms: convertFilterAtoms(filterAtoms),
        geoIdFilter,
        foundHotelCount,
        quickFilters,
        quickControls,
        detailedFilters,
        detailedFiltersBatches,
    };

    return {
        permanentFilters: filters,
        currentFilters: filters,
    };
};

export const getPermanentAndCurrentFilters = ({
    targetFiltersType,
    state,
    getUpdatedFilters,
}: {
    targetFiltersType: IChangeFiltersTargetType['targetFiltersType'];
    state: IHotelsFiltersReducer;
    getUpdatedFilters: (filters: IFullFilters) => IFullFilters;
}): IHotelsFiltersReducer => {
    const {currentFilters} = state;
    const updatedFilters = getUpdatedFilters(currentFilters);

    switch (targetFiltersType) {
        case 'ALL': {
            return {
                ...state,
                needSyncPermanentFiltersWithServer: true,
                permanentFilters: updatedFilters,
                currentFilters: updatedFilters,
            };
        }
        case 'CURRENT': {
            return {
                ...state,
                needSyncCurrentFiltersWithServer: true,
                currentFilters: updatedFilters,
            };
        }
    }
};

export const getUpdatedFiltersByActiveAtoms =
    (payload: IChangeFilterGroupPayload) =>
    (filters: IFullFilters): IFullFilters => {
        const {atomsOn, atomsOff} = payload;
        const {activeFilterAtoms} = filters;
        const updatedActiveFilterAtoms = {
            ...activeFilterAtoms,
        };

        if (atomsOn) {
            atomsOn.forEach((atom: TFilterAtomType) => {
                updatedActiveFilterAtoms[atom] = true;
            });
        }

        if (atomsOff) {
            atomsOff.forEach((atom: TFilterAtomType) => {
                delete updatedActiveFilterAtoms[atom];
            });
        }

        return {
            ...filters,
            activeFilterAtoms: updatedActiveFilterAtoms,
        };
    };

export const getUpdatedFiltersByResetAction = (
    payload: IResetFilterAction,
    filters: IFullFilters,
) => {
    let newPriceFilter = null;
    let newGeoIdFilter = null;
    const {atomsOff, needResetFilterPrice, needResetGeoIdFilter} = payload;
    const {activeFilterAtoms} = filters;
    const updatedActiveFilterAtoms = {
        ...activeFilterAtoms,
    };

    if (needResetFilterPrice && filters.priceFilter) {
        newPriceFilter = getResetPriceFilter(filters.priceFilter);
    }

    if (needResetGeoIdFilter && filters.geoIdFilter) {
        newGeoIdFilter = {
            ...filters.geoIdFilter,
            selected: false,
        };
    }

    if (atomsOff) {
        atomsOff.forEach((atom: TFilterAtomType) => {
            delete updatedActiveFilterAtoms[atom];
        });
    }

    return {
        ...filters,
        activeFilterAtoms: updatedActiveFilterAtoms,
        ...(newPriceFilter ? {priceFilter: newPriceFilter} : {}),
        ...(newGeoIdFilter ? {geoIdFilter: newGeoIdFilter} : {}),
    };
};

export const getUpdatedFiltersCurrentGeoId =
    (payload: ICurrentGeoIdFilterPayloadWithTarget) =>
    (filters: IFullFilters): IFullFilters => {
        const {geoIdFilter} = filters;
        const {selected} = payload;

        if (geoIdFilter) {
            return {
                ...filters,
                geoIdFilter: {
                    ...geoIdFilter,
                    selected,
                },
            };
        }

        return filters;
    };

export const getUpdatedFiltersRangePrice =
    (payload: IChangePriceFilterPayload) =>
    (filters: IFullFilters): IFullFilters => {
        const {priceFilter} = filters;

        if (priceFilter) {
            return {
                ...filters,
                priceFilter: {
                    ...priceFilter,
                    ...payload,
                },
            };
        }

        return filters;
    };

export const resetFiltersRangePrice = (filters: IFullFilters) => ({
    ...filters,
    priceFilter: getResetPriceFilter(filters.priceFilter),
});

export const getResetPriceFilter = (value?: IRangePriceFilter) =>
    value && {
        ...value,
        minValue: value.minPriceEstimate,
        maxValue: value.maxPriceEstimate,
    };

export const getCurrentFiltersAfterReset = (
    currentFilters: IFullFilters,
): IFullFilters => ({
    ...currentFilters,
    activeFilterAtoms: {},
    priceFilter: getResetPriceFilter(currentFilters.priceFilter),
});
