import {createReducer, ActionType} from 'typesafe-actions';
import _uniq from 'lodash/uniq';

import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {
    INavigationTokens,
    NavigationTokenType,
} from 'types/hotels/search/INavigationTokens';

import * as actions from './actions';

export interface IHotelListReducer {
    navigationToken?: NavigationTokenType;
    navigationTokens?: INavigationTokens;
    hotels: PermalinkType[];
}

/* Initial State */

const INITIAL_STATE: IHotelListReducer = {
    hotels: [],
};

/* Reducer */

const hotelListReducer = createReducer<
    IHotelListReducer,
    ActionType<typeof actions>
>(INITIAL_STATE)
    .handleAction(
        actions.setNavigationToken,
        (state, {payload}): IHotelListReducer => ({
            ...state,
            navigationToken: payload,
        }),
    )
    .handleAction(
        actions.fillHotelList,
        (state, {payload}): IHotelListReducer => {
            const {hotels: hotelsFromState} = state;

            const {pollingResponse, pageHotelCount, withInfiniteScroll} =
                payload;
            const {
                hotels: hotelsBySearch,
                pricedHotelCount,
                navigationTokens,
            } = pollingResponse;
            const hotelsBySearchWithPrice = hotelsBySearch
                .slice(0, pricedHotelCount)
                .map(({hotel}) => hotel.permalink);
            let hotels = _uniq([
                ...hotelsFromState,
                ...hotelsBySearchWithPrice,
            ]);

            if (!withInfiniteScroll) {
                hotels = hotels.slice(0, pageHotelCount);
            }

            const {currentPage: navigationToken} = navigationTokens;

            return {
                ...state,
                hotels,
                navigationToken,
                navigationTokens,
            };
        },
    )
    .handleAction(actions.resetHotelList, (state): IHotelListReducer => {
        return {
            ...state,
            hotels: [],
        };
    })
    .handleAction(
        actions.resetHotelListToInitial,
        (): IHotelListReducer => INITIAL_STATE,
    );

export default hotelListReducer;
