import {createReducer} from 'typesafe-actions';
import _keyBy from 'lodash/keyBy';
import _getFP from 'lodash/fp/get';
import _pickBy from 'lodash/pickBy';

import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';
import {
    IHotelOfferSearchProgress,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {
    THotelWithOffersByPermalink,
    IHotelWithOffers,
} from 'types/hotels/hotel/IHotelWithOffers';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';
import {EStartSearchReason} from 'types/hotels/search/ISearchReason';
import {IActualGeoRegion} from 'types/hotels/hotel/IActualGeoRegion';
import {HotelSlugType} from 'types/hotels/hotel/IHotel';

import {
    clearHotels,
    clearNotFinishedHotels,
    clearSearchResult,
    searchHotelsActions,
    SearchHotelsActionsType,
    clearSearchBeforeNewSearch,
    changeHotelsIsFavoriteOnSearchActions,
    changeIsOpenForNoMoreFavoritesModalAction,
    clearLastSearchTimestampAction,
} from './actions';

export enum ESearchBanner {
    MIR = 'MIR',
}

export interface ISearchHotelsData {
    context: string;
    pollEpoch: number;
    pollIteration: number;
    foundHotelCount: number;
    pricedHotelCount: number;
    actualRegion: IActualGeoRegion;
    searchRegion: IGeoRegion;
    offerSearchParams: IRequiredOfferParams;
    offerSearchProgress: IHotelOfferSearchProgress;
    hotelWithOffersByPermalink: THotelWithOffersByPermalink;
    operatorById: OperatorByIdType;
    searchBannerType?: ESearchBanner;
    topHotelSlug?: HotelSlugType;
    searchPagePollingId?: string;
    hasBoyOffers: boolean;
}

export interface ISearchHotelsReducer {
    isLoading: boolean;
    isError: boolean;
    lastSearchTimestamp: number | null;
    startSearchReason: EStartSearchReason;
    data?: ISearchHotelsData;
    startSearchTime?: number;
    noMoreFavoritesModalIsOpen?: boolean;
}

/* Initial State */

const initialState: ISearchHotelsReducer = {
    isLoading: true,
    isError: false,
    lastSearchTimestamp: null,
    startSearchReason: EStartSearchReason.MOUNT,
    noMoreFavoritesModalIsOpen: false,
};

/* Reducer */

const searchHotelsReducer = createReducer<
    ISearchHotelsReducer,
    SearchHotelsActionsType
>(initialState)
    .handleAction(
        searchHotelsActions.request,
        (state, {payload: {startSearchReason}}): ISearchHotelsReducer => ({
            ...state,
            isLoading: true,
            startSearchReason,
        }),
    )
    .handleAction(
        searchHotelsActions.success,
        (state, {payload}): ISearchHotelsReducer => {
            const {data} = state;
            const {
                context,
                pollEpoch,
                pollIteration,
                foundHotelCount,
                pricedHotelCount,
                actualRegion,
                hotels,
                searchRegion,
                offerSearchParams,
                offerSearchProgress,
                operatorById,
                searchBannerType,
                topHotelSlug,
                searchPagePollingId,
                hasBoyOffers,
            } = payload;

            const hotelByPermalinkBySearch = _keyBy(
                hotels,
                _getFP<IHotelWithOffers, 'hotel', 'permalink'>([
                    'hotel',
                    'permalink',
                ]),
            );
            const hotelByPermalinkFromState = _getFP<
                ISearchHotelsData,
                'hotelWithOffersByPermalink'
            >(['hotelWithOffersByPermalink'])(data);
            const operatorByIdFromState = _getFP<
                ISearchHotelsData,
                'operatorById'
            >(['operatorById'])(data);

            return {
                ...state,
                isLoading: false,
                isError: false,
                lastSearchTimestamp: Date.now(),
                data: {
                    context,
                    pollEpoch,
                    pollIteration,
                    foundHotelCount,
                    pricedHotelCount,
                    actualRegion,
                    searchRegion,
                    offerSearchParams,
                    offerSearchProgress,
                    operatorById: {
                        ...operatorByIdFromState,
                        ...operatorById,
                    },
                    hotelWithOffersByPermalink: {
                        ...hotelByPermalinkFromState,
                        ...hotelByPermalinkBySearch,
                    },
                    searchBannerType,
                    topHotelSlug,
                    searchPagePollingId,
                    hasBoyOffers,
                },
            };
        },
    )
    .handleAction(
        searchHotelsActions.failure,
        (state): ISearchHotelsReducer => ({
            ...state,
            isError: true,
        }),
    )
    .handleAction(clearHotels, (state): ISearchHotelsReducer => {
        const {data} = state;

        if (data) {
            return {
                ...state,
                data: {
                    ...data,
                    hotelWithOffersByPermalink: {},
                },
            };
        }

        return state;
    })
    .handleAction(clearNotFinishedHotels, (state): ISearchHotelsReducer => {
        const {data} = state;

        if (data) {
            const {hotelWithOffersByPermalink} = data;
            const clearedHotelWithOffersByPermalink = _pickBy(
                hotelWithOffersByPermalink,
                ({searchIsFinished}) => searchIsFinished,
            );

            return {
                ...state,
                data: {
                    ...data,
                    hotelWithOffersByPermalink:
                        clearedHotelWithOffersByPermalink,
                },
            };
        }

        return state;
    })
    .handleAction(clearSearchResult, (): ISearchHotelsReducer => initialState)
    .handleAction(
        clearSearchBeforeNewSearch,
        (state): ISearchHotelsReducer => ({
            ...initialState,
            startSearchReason: state.startSearchReason,
            startSearchTime: Date.now(),
        }),
    )
    .handleAction(
        clearLastSearchTimestampAction,
        (state): ISearchHotelsReducer => ({
            ...state,
            lastSearchTimestamp: null,
        }),
    )
    .handleAction(
        changeHotelsIsFavoriteOnSearchActions.success,
        (state, {payload}): ISearchHotelsReducer => {
            const {data} = state;
            const isFavoriteSelectedHotelByPermalink =
                data?.hotelWithOffersByPermalink[payload].hotel.isFavorite;
            const selectedHotelByPermalink = {
                ...data?.hotelWithOffersByPermalink[payload].hotel,
                isFavorite: !isFavoriteSelectedHotelByPermalink,
            };

            return {
                ...state,
                data: {
                    ...state.data,
                    hotelWithOffersByPermalink: {
                        ...state.data?.hotelWithOffersByPermalink,
                        [payload]: {
                            ...state.data?.hotelWithOffersByPermalink[payload],
                            hotel: selectedHotelByPermalink,
                        },
                    },
                } as ISearchHotelsData,
            };
        },
    )
    .handleAction(
        changeIsOpenForNoMoreFavoritesModalAction,
        (state, {payload}): ISearchHotelsReducer => {
            return {
                ...state,
                noMoreFavoritesModalIsOpen: payload,
            };
        },
    );

export default searchHotelsReducer;
