import isEqual from 'lodash/isEqual';

import {
    LT_DEFAULT_SEARCH_CITY,
    LT_DEFAULT_SUGGESTS_COUNT,
} from 'projects/index/IndexApp/components/IndexTours/constants';

import {Request} from '@yandex-data-ui/core/lib/types';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {
    ESuggestDestinationType,
    IToursSearchFormFromPointField,
    IToursSearchFormToPointField,
} from 'projects/index/IndexApp/components/IndexTours/components/SearchForm/types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    getSearchSuggestsDestinationActions,
    getSearchSuggestsDepartureActions,
    setDepartures,
} from 'reducers/tours/searchSuggests/actions';
import {setToursSearchFormFromFieldAction} from 'reducers/tours/searchForm/actions';

import {getUserGeoLocation} from 'selectors/common/userInfoSelector';

import {logError} from 'utilities/logger/logError';
import findMatchedDeparture from 'utilities/tours/findMatchedDeparture';
import parseAutocompleteResponse from 'utilities/tours/parseAutocompleteResponse';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

import {levelTravel} from 'serviceProvider/tours/levelTravelProvider/levelTravelProvider';

export function fillSearchSuggestFromByGeobase({
    req,
}: {
    req?: Request;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const state = getState();

        const geoLocation = getUserGeoLocation(state);

        const departuresResponse = await levelTravel
            .provider(req?.container)
            .getDepartures({prioritized_count: 10});

        const matchedGeo = findMatchedDeparture({
            departures: departuresResponse.departures,
            departureName: geoLocation.geoName || '',
        });

        if (matchedGeo) {
            dispatch(
                setToursSearchFormFromFieldAction({
                    selectedValue: matchedGeo,
                    inputValue: matchedGeo.name_ru,
                    source: ESuggestSource.SEARCH_CONTEXT,
                }),
            );
        }
    };
}

export function requestSearchSuggestsFromThunkAction({
    req,
    query,
}: {
    req?: Request;
    query: string;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const state = getState();

        dispatch(
            getSearchSuggestsDepartureActions.request({
                requestParams: {query},
                fieldType: ESearchFormFieldName.FROM,
            }),
        );

        const departuresResponse = state.tours.searchSuggests.departures?.length
            ? state.tours.searchSuggests.departures
            : (await levelTravel.provider(req?.container).getDepartures({}))
                  ?.departures;

        if (
            !state.tours.searchSuggests.departures?.length &&
            departuresResponse
        ) {
            dispatch(setDepartures(departuresResponse));
        }

        dispatch(
            getSearchSuggestsDepartureActions.success({
                fieldType: ESearchFormFieldName.FROM,
                items: departuresResponse
                    .filter(
                        departure =>
                            !query ||
                            departure.name_ru
                                .toLocaleLowerCase()
                                .includes(query) ||
                            departure.name_en
                                .toLocaleLowerCase()
                                .includes(query),
                    )
                    .slice(0, LT_DEFAULT_SUGGESTS_COUNT),
            }),
        );
    };
}

export function requestSearchToSuggestsThunkAction({
    req,
    params,
}: {
    req?: Request;
    params: {query: string; from_city: string};
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const state = getState();

        dispatch(
            getSearchSuggestsDestinationActions.request({
                fieldType: ESearchFormFieldName.TO,
                requestParams: {
                    query: params.query,
                    from_city: params.from_city,
                },
            }),
        );

        if (!params.query) {
            const currentSuggest =
                state.tours.searchForm[ESearchFormFieldName.FROM].selectedValue;
            const countries = await levelTravel
                .provider(req?.container)
                .getCountries({
                    from_city: currentSuggest
                        ? currentSuggest.name_en
                        : params.from_city || LT_DEFAULT_SEARCH_CITY,
                });

            dispatch(
                getSearchSuggestsDestinationActions.success({
                    fieldType: ESearchFormFieldName.TO,
                    items:
                        countries
                            ?.slice(0, LT_DEFAULT_SUGGESTS_COUNT)
                            .map(country => ({
                                ...country,
                                type: ESuggestDestinationType.COUNTRY,
                                visaType: country.visa_type_ru,
                            })) || [],
                }),
            );
        }

        const autocomplete = await levelTravel
            .provider(req?.container)
            .autocomplete({
                query: params.query,
                from_city: params.from_city,
            });

        if (autocomplete && autocomplete.length) {
            const items = parseAutocompleteResponse(autocomplete);

            dispatch(
                getSearchSuggestsDestinationActions.success({
                    fieldType: ESearchFormFieldName.TO,
                    items: items.slice(0, LT_DEFAULT_SUGGESTS_COUNT),
                }),
            );
        }
    };
}

export default function requestSearchSuggestsThunkAction({
    fromField,
    toField,
    req,
}: {
    fromField: IToursSearchFormFromPointField;
    toField: IToursSearchFormToPointField;
    req?: Request;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const state = getState();

            const {
                searchSuggests: {
                    from: {requestParams: prevFromSuggestsRequestParams},
                    to: {requestParams: prevToSuggestsRequestParams},
                },
            } = state.tours;

            const fromSuggestsRequestParams = {
                query: (fromField.inputValue || '').toLowerCase(),
            };

            if (
                !isEqual(
                    prevFromSuggestsRequestParams,
                    fromSuggestsRequestParams,
                )
            ) {
                dispatch(
                    requestSearchSuggestsFromThunkAction({
                        req,
                        query: fromSuggestsRequestParams.query,
                    }),
                );
            }

            const currentSuggest =
                state.tours.searchForm[ESearchFormFieldName.FROM].selectedValue;

            const toSuggestsRequestParams = {
                query: (toField.inputValue || '').toLowerCase(),
                from_city: currentSuggest
                    ? currentSuggest.name_en
                    : LT_DEFAULT_SEARCH_CITY,
            };

            if (
                !isEqual(prevToSuggestsRequestParams, toSuggestsRequestParams)
            ) {
                dispatch(
                    requestSearchToSuggestsThunkAction({
                        req,
                        params: toSuggestsRequestParams,
                    }),
                );
            }
        } catch (err) {
            logError({message: 'LevelTrevelError'}, err);
        }
    };
}
