import {
    CommonSearchError,
    OutdatedSearchError,
    PointNotFoundSearchError,
} from 'projects/trains/lib/search/constants';

import {IRaspParseContextServiceParams} from 'server/services/RaspService/types/IRaspParseContextServiceParams';
import {
    isFilledContextPoint,
    ITrainsSearchContextWithoutSearchInfo,
} from 'reducers/trains/context/types';
import {EDirection} from 'types/common/EDirection';
import {ETransport} from 'types/common/ETransport';
import {TTrainsSegmentId} from 'types/trains/common/segment/ITrainsSegment';

import {CustomThunkAction} from 'reducers/trains/customDispatch';

import {ROBOT} from 'utilities/dateUtils/formats';
import {
    getDateFromSpecialValue,
    isWhenSpecialValue,
} from 'projects/trains/lib/context';
import isTrainsOutdatedOrWrongWhen from 'projects/trains/lib/date/isTrainsOutdatedOrWrongWhen';
import {checkTrainsInvalidReturnWhen} from 'projects/trains/lib/date/checkTrainsInvalidReturnWhen';

import {IDependencies} from 'server/getContainerConfig';

import {raspService} from 'serviceProvider';

import {setTrainsContext} from '../actions';

export interface ITrainsParseContextParams {
    when?: string;
    originalWhen?: string;
    returnWhen?: string;
    from: {
        slug?: string;
        key?: string;
        title?: string;
    };
    to: {
        slug?: string;
        key?: string;
        title?: string;
    };
    direction?: EDirection;
    forwardSegmentId?: TTrainsSegmentId;
    backwardSegmentId?: TTrainsSegmentId;
}

/**
 * Изоморфный экшен для получения информации о пунктах отправления и прибытия.
 *
 * Пробрасывает дальше ошибки контекста, если такие были. Например, если не найден какой-либо из пунктов.
 *
 * @param context - поисковый контекст
 * @param [container] - DI контейнер
 */
export default function trainsParseContext(
    context: ITrainsParseContextParams,
    container?: IDependencies,
): CustomThunkAction<void> {
    return async dispatch => {
        let parsedContext;

        try {
            const params: IRaspParseContextServiceParams = {
                transport: ETransport.TRAIN,
            };

            if (context.from.slug) {
                params.from_slug = context.from.slug;
            } else {
                params.from_key = context.from.key;
                params.from_title = context.from.title;
            }

            if (context.to.slug) {
                params.to_slug = context.to.slug;
            } else {
                params.to_key = context.to.key;
                params.to_title = context.to.title;
            }

            parsedContext = await raspService
                .provider(container)
                .parseContext(params);
        } catch (err) {
            throw new CommonSearchError();
        }

        if (!parsedContext) {
            throw new CommonSearchError();
        }

        let newWhen = context.when;

        // @see https://st.yandex-team.ru/TRAVELFRONT-1691
        if (
            context.originalWhen &&
            isWhenSpecialValue(context.originalWhen) &&
            isFilledContextPoint(parsedContext.from)
        ) {
            const date = getDateFromSpecialValue(context.originalWhen);

            if (date) {
                newWhen = date.tz(parsedContext.from.timezone).format(ROBOT);
            }
        }

        const contextToSet: ITrainsSearchContextWithoutSearchInfo = {
            when: newWhen || '',
            originalWhen: context.originalWhen || '',
            returnWhen: context.returnWhen || null,
            direction: context.direction || EDirection.FORWARD,
            forwardSegmentId: context.forwardSegmentId || null,
            backwardSegmentId: context.backwardSegmentId || null,
            latestDatetime: null,
            isChanged: null,
            original: null,
            ...parsedContext,
        };

        dispatch(setTrainsContext(contextToSet));

        /*
         * Выкидываем ошибку после того, как записали поисковый контекст,
         * чтобы в поисковой форме были заполнены поля с вадидными пунктами.
         */
        if (
            parsedContext.errors.some(error => error.type === 'point_not_found')
        ) {
            throw new PointNotFoundSearchError();
        }

        // Делаем проверку на протухший поиск после того как сохраним контекст,
        // т.к. он нам понадобится в случае редиректа (чтобы понять, куда редиректить)
        const {from} = parsedContext;

        if (!isFilledContextPoint(from)) {
            return;
        }

        if (newWhen && isTrainsOutdatedOrWrongWhen(newWhen, from.timezone)) {
            throw new OutdatedSearchError();
        }

        if (checkTrainsInvalidReturnWhen(newWhen, context.returnWhen)) {
            throw new OutdatedSearchError();
        }
    };
}
