import {
    IRaspParseContextApiResponse,
    IFilledContextPoint,
    TContextPointType,
} from 'server/api/RaspApi/types/IRaspParseContextApiResponse';
import {TTrainsSegmentId} from 'types/trains/common/segment/ITrainsSegment';
import {EDirection} from 'types/common/EDirection';
import {WHEN_SPECIAL_VALUE} from 'types/common/When';

/*
 * Поисковый контекст в основном содержит в себе информацию по пунктам отправления и прибытия.
 * Эту информацию можно разделить на два блока, с разными источниками.
 *
 * 1. Основная информация по пунктам прибытия и отправления + время поиска.
 *    Приходит из ручки parseContext. Одновременно с ответом ручки на фронте заполняем поля when и originalWhen.
 *
 * 2. Информация, более специфичная для поиска.
 *    Приходит из ручки search. Содержит поля latestDatetime, isChanged, original.
 *
 * Соответственно возможно несколько конфигурациий контекста.
 *
 * 1. ITrainsSearchContextWithoutSearchInfo - получили ответ ручки parseContext, без ручки search.
 * 2. ITrainsFilledSearchContext - точно есть ответ ручки parseContext с проверкой,
 *    что есть информация о пунктах отправления и прибытия (если информации нет, то отображаем 404 страницу).
 *    Также возможен ответ ручки search.
 * 3. TrainsSearchContextType - то, что хранится в store. ITrainsFilledSearchContext + для всех свойств
 *    дефолтные значения null.
 */

/**
 * Информация о пункте прибытия/отправления из ручки search
 */
export interface ITrainsSearchContextPoint {
    key: string;
    title: string;
    titleWithType?: string;
    shortTitle?: string;
    titleGenitive?: string;
    titleLocative?: string;
    preposition?: string;
    popularTitle?: string;
    titleAccusative?: string;
}

export function isFilledContextPoint(
    point: TContextPointType,
): point is IFilledContextPoint {
    return Boolean(
        point.timezone && point.region && point.settlement && point.country,
    );
}

export interface ITrainsFilledContextPoints {
    from: IFilledContextPoint;
    to: IFilledContextPoint;
}

export interface ITrainsSearchContextWithoutSearchInfo
    extends IRaspParseContextApiResponse {
    when: string;
    originalWhen: WHEN_SPECIAL_VALUE | string;
    returnWhen: string | null;
    direction?: EDirection | null;
    forwardSegmentId?: TTrainsSegmentId | null;
    backwardSegmentId?: TTrainsSegmentId | null;
    latestDatetime: null;
    isChanged: null;
    original: null;
}

export interface ITrainsFilledSearchContext
    extends Omit<
        ITrainsSearchContextWithoutSearchInfo,
        | 'from'
        | 'to'
        | 'originalFrom'
        | 'originalTo'
        | 'latestDatetime'
        | 'isChanged'
        | 'original'
    > {
    from: IFilledContextPoint;
    to: IFilledContextPoint;
    originalFrom: IFilledContextPoint;
    originalTo: IFilledContextPoint;
    latestDatetime: string | null;
    isChanged: boolean | null;
    original: {
        pointFrom: ITrainsSearchContextPoint;
        pointTo: ITrainsSearchContextPoint;
    } | null;
}

export function isFilledTrainsSearchContext(
    context: TrainsSearchContextType,
): context is ITrainsFilledSearchContext {
    return Boolean(
        context.from &&
            context.from.timezone &&
            context.to &&
            context.to.timezone,
    );
}

export type TrainsSearchContextType = WithNullValues<
    Omit<
        ITrainsFilledSearchContext,
        'from' | 'to' | 'originalFrom' | 'originalTo'
    > & {
        from: TContextPointType;
        to: TContextPointType;
    }
>;

export interface ITrainsSearchApiContext {
    transportTypes: string[];
    search: {
        nearest: boolean;
        pointTo: ITrainsSearchContextPoint;
        pointFrom: ITrainsSearchContextPoint;
    };
    original: {
        nearest: boolean;
        pointTo: ITrainsSearchContextPoint;
        pointFrom: ITrainsSearchContextPoint;
    };
    latestDatetime: string;
    isChanged: boolean;
}
