import {
    CommonSearchError,
    ESearchErrorTypes,
    OutdatedSearchError,
    PointNotFoundSearchError,
    SamePointsSearchError,
} from 'projects/trains/lib/search/constants';

import {Request} from '@yandex-data-ui/core/lib/types';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';

import {
    trainsSearchActions,
    clearTrainsSearchResult,
} from 'reducers/trains/genericSearch/search/actions';
import {fillFormFromContextThunkAction} from 'reducers/trains/searchForm/thunk/fillFormFromContextThunkAction';
import {updatePurchaseAvailability} from 'reducers/trains/actions/updatePurchaseAvailability';
import {CustomThunkAction} from 'reducers/trains/customDispatch';
import trainsParseContext, {
    ITrainsParseContextParams,
} from 'reducers/trains/context/actions/trainsParseContext';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';

import scrollTo from 'utilities/dom/scrollTo';
import {checkContextSamePointsError} from 'projects/trains/lib/context/checkSamePointsContextError';

function preFetchDataForGenericSearch({
    context,
    req,
    canStartSearch,
}: {
    context: ITrainsParseContextParams;
    req?: Request;
    canStartSearch?: boolean;
}): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        const container = req?.container;

        try {
            dispatch(clearTrainsSearchResult());

            await dispatch(trainsParseContext(context, container));

            dispatch(fillFormFromContextThunkAction());

            await dispatch(updatePurchaseAvailability(container));

            const searchContext = trainsContextSelector(getState());

            checkContextSamePointsError(searchContext);

            if (canStartSearch) {
                if (
                    isFilledTrainsSearchContext(searchContext) &&
                    !searchContext.errors?.length
                ) {
                    dispatch(trainsSearchActions.request(undefined));
                }
            }
        } catch (e) {
            switch (true) {
                case e instanceof CommonSearchError: {
                    dispatch(
                        trainsSearchActions.failure({
                            errorCode: ESearchErrorTypes.COMMON,
                        }),
                    );

                    break;
                }

                case e instanceof PointNotFoundSearchError: {
                    dispatch(
                        trainsSearchActions.failure({
                            errorCode: ESearchErrorTypes.POINT_NOT_FOUND,
                        }),
                    );

                    break;
                }

                case e instanceof SamePointsSearchError: {
                    dispatch(
                        trainsSearchActions.failure({
                            errorCode: ESearchErrorTypes.SAME_POINTS,
                        }),
                    );

                    break;
                }

                case e instanceof OutdatedSearchError: {
                    dispatch(
                        trainsSearchActions.failure({
                            errorCode: ESearchErrorTypes.OUT_DATE,
                        }),
                    );

                    if (__SERVER__) {
                        throw e;
                    }

                    break;
                }
            }

            dispatch(fillFormFromContextThunkAction());
        }
    };
}

export function preFetchDataForGenericSearchServer({
    context,
    req,
}: {
    context: ITrainsParseContextParams;
    req: Request;
}): CustomThunkAction<void> {
    return async (dispatch): Promise<void> => {
        await dispatch(preFetchDataForGenericSearch({context, req}));
    };
}

export function preFetchDataForGenericSearchClient(
    context: ITrainsParseContextParams,
): CustomThunkAction<void> {
    return async (dispatch): Promise<void> => {
        scrollTo({top: 0, left: 0});
        await dispatch(
            preFetchDataForGenericSearch({context, canStartSearch: true}),
        );
    };
}
