import {getType, isActionOf} from 'typesafe-actions';
import findLastIndex from 'lodash/findLastIndex';

import {
    PASSENGERS_TYPES,
    PASSENGERS_TYPE_ORDER,
} from 'projects/trains/constants/passengersTypes';

import {ITrainPassenger} from 'reducers/trains/order/types';

import {setCount} from 'reducers/trains/order/actions/passengers';
import * as bonusCardsActions from 'reducers/trains/order/actions/bonusCards';
import * as passengerDocumentActions from 'reducers/trains/order/actions/passengerDocument';
import * as singlePassengerDataActions from 'reducers/trains/order/actions/singlePassengerData';
import {singlePassengerData} from 'reducers/trains/order/reducers/singlePassengerData';
import {PASSENGERS_DEFAULT_STATE} from 'reducers/trains/order/reducers/passengers';
import {ProjectAction} from 'reducers/storeTypes';

const passengersDataActions = [
    ...Object.values(bonusCardsActions),
    ...Object.values(passengerDocumentActions),
    ...Object.values(singlePassengerDataActions),
];

const passengersCountActions = Object.values(setCount);

const defaultFieldValue = {value: ''};
const emptyPassenger: ITrainPassenger = {
    lastName: defaultFieldValue,
    firstName: defaultFieldValue,
    patronymic: defaultFieldValue,
    passengerDocument: {
        type: {value: null},
        number: {value: ''},
        country: {value: null},
        validDate: {value: ''},
    },
    gender: {value: null},
    birthDate: defaultFieldValue,
    emailOrPhone: defaultFieldValue,
    hasDiscountDocument: false,
    ageGroup: PASSENGERS_TYPES.ADULTS,
    bonusCards: {},
    isNonRefundableTariff: false,
};

const actionAgeMap = {
    [getType(setCount[PASSENGERS_TYPES.ADULTS])]: PASSENGERS_TYPES.ADULTS,
    [getType(setCount[PASSENGERS_TYPES.CHILDREN])]: PASSENGERS_TYPES.CHILDREN,
    [getType(setCount[PASSENGERS_TYPES.BABIES])]: PASSENGERS_TYPES.BABIES,
};

const DEFAULT_STATE = PASSENGERS_TYPE_ORDER.reduce(
    (acc: ITrainPassenger[], type) => [
        ...acc,
        ...Array.from({length: PASSENGERS_DEFAULT_STATE[type]}, () =>
            createPassenger(type),
        ),
    ],
    [],
);

function createPassenger(type: PASSENGERS_TYPES): ITrainPassenger {
    return [singlePassengerDataActions.setAgeGroup(type, 0)].reduce(
        (state, action) => singlePassengerData(state, action),
        emptyPassenger,
    );
}

function addPassengers(
    passengers: ITrainPassenger[],
    type: PASSENGERS_TYPES,
    count: number,
): ITrainPassenger[] {
    const groupIndex = PASSENGERS_TYPE_ORDER.indexOf(type);

    const passengerIndex =
        findLastIndex(
            passengers,
            (pass: ITrainPassenger) =>
                PASSENGERS_TYPE_ORDER.indexOf(pass.ageGroup) <= groupIndex,
        ) + 1;
    const result = [...passengers];

    result.splice(
        passengerIndex,
        0,
        ...Array.from({length: count}, () => createPassenger(type)),
    );

    return result;
}

function removePassengers(
    passengers: ITrainPassenger[],
    type: PASSENGERS_TYPES,
    count: number,
): ITrainPassenger[] {
    return passengers.reduceRight(
        (result: ITrainPassenger[], item: ITrainPassenger) => {
            if (item.ageGroup !== type || count <= 0) {
                return [item, ...result];
            }

            count--;

            return result;
        },
        [],
    );
}

function changePassengersCount(
    passengers: ITrainPassenger[],
    type: PASSENGERS_TYPES,
    count: number,
): ITrainPassenger[] {
    if (count < 0) {
        return removePassengers(passengers, type, -count);
    } else if (count > 0) {
        return addPassengers(passengers, type, count);
    }

    return passengers;
}

export default function passengersData(
    state?: ITrainPassenger[],
    action?: ProjectAction,
): ITrainPassenger[] {
    if (!action || !state) {
        return DEFAULT_STATE;
    }

    if (isActionOf(passengersDataActions, action)) {
        const {meta: index} = action;

        if (index >= state.length || index < 0) {
            return state;
        }

        return state.map((passenger, passengerIndex) => {
            if (passengerIndex !== index) {
                return passenger;
            }

            return singlePassengerData(state[index], action);
        });
    }

    if (isActionOf(passengersCountActions, action)) {
        const ageType = actionAgeMap[action.type];
        const passengersCountByAgeType = state.filter(
            passenger => passenger.ageGroup === ageType,
        ).length;
        const diff = action.payload - passengersCountByAgeType;

        return changePassengersCount(state, ageType, diff);
    }

    return state;
}
