import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {SOLOMON_TRAINS_PAY_BUTTON_CLICK} from 'constants/solomon';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {EDirection} from 'types/common/EDirection';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setDirection,
    setIndex,
    setOrderStep,
} from 'reducers/trains/order/actions/view';

import {reachGoalOnce} from 'utilities/metrika';
import history from 'utilities/browserHistory/browserHistory';
import needToCancelOrder from 'projects/trains/lib/order/needToCancelOrder';
import {trainsURLs} from 'projects/trains/lib/urls';
import {sendPaymentCounter} from 'utilities/solomon';
import getFirstForwardSegment from 'projects/trains/lib/complexOrder/getFirstForwardSegment';
import getFirstBackwardSegment from 'projects/trains/lib/complexOrder/getFirstBackwardSegment';

export type TOrderPlaceStepDescription = {
    step: ORDER_STEP.PLACES;
    index: number;
    direction: EDirection;
};
export type TOtherStepDescription = {
    step: ORDER_STEP.PASSENGERS | ORDER_STEP.CONFIRM | ORDER_STEP.PAYMENT;
};
export type TOrderStepDescription =
    | TOrderPlaceStepDescription
    | TOtherStepDescription;

export default function changeOrderStep(
    stepDescription: TOrderStepDescription,
): CustomThunkAction<void> {
    return function (dispatch, getState): void {
        const {
            trains: {context, order},
        } = getState();
        const segment = getFirstForwardSegment(order);
        const backwardSegment = getFirstBackwardSegment(order);

        if (!segment || !isFilledTrainsSearchContext(context)) {
            return;
        }

        if (
            needToCancelOrder({
                ...order,
                orderStep: stepDescription.step,
            })
        ) {
            reachGoalOnce(ETrainsGoal.CANCEL_ORDER_BY_TAB);
        }

        dispatch(setOrderStep(stepDescription.step));

        if (stepDescription.step === ORDER_STEP.PLACES) {
            const {direction, index} = stepDescription;

            dispatch(setDirection(direction));
            dispatch(setIndex(index));
        }

        if (stepDescription.step === ORDER_STEP.PAYMENT) {
            sendPaymentCounter(SOLOMON_TRAINS_PAY_BUTTON_CLICK);
        }

        if (history) {
            const {location} = history;

            const currentUrl = `${location.pathname}${location.search}`;

            const newOrderUrl = trainsURLs.getTrainOrderUrl({
                context,
                segment,
                backwardSegment,
                order,
                orderStepDescription: stepDescription,
            });

            if (newOrderUrl !== currentUrl) {
                history.push(newOrderUrl);
            } else if (
                [ORDER_STEP.CONFIRM, ORDER_STEP.PAYMENT].includes(
                    order.orderStep,
                ) &&
                [ORDER_STEP.PLACES, ORDER_STEP.PASSENGERS].includes(
                    stepDescription.step,
                )
            ) {
                // При переходе назад с шагов подтверждения или оплаты на шаг выбора мест или ввода данных,
                // заменяем в истории браузера шаг с подтверждением данных/оплаты,
                // чтобы нельзя было вернуться с помощью браузерного вперед.
                history.push(currentUrl);
                history.goBack();
            }
        }
    };
}
