import {batchActions} from 'redux-batched-actions';
import flatMap from 'lodash/flatMap';
import times from 'lodash/times';
import {Action} from 'redux';

import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {DIRECTIONS, EDirection} from 'types/common/EDirection';
import {ITrainsDirectionAndIndex} from 'types/trains/common/segment/ITrainsDirectionAndIndex';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {setOrderPageStatus} from 'reducers/trains/order/actions/flow';
import {
    clearPlaces,
    clearRequirements,
    setCoach,
    setOrderCoachCategory,
    setOrderCoachType,
    setPreferredGender,
} from 'reducers/trains/order/actions/trains';
import {setOrderError} from 'reducers/trains/order/actions/view';
import changeOrderStep from 'reducers/trains/order/thunk/changeOrderStep';
import {requestAllTrainDetails} from 'reducers/trains/order/thunk/requestAllTrainDetails';
import clearOrder from 'reducers/trains/order/thunk/clearOrder';

import directionTrainsCountSelector from 'selectors/trains/order/directionTrainsCountSelector';

import {getTrainsOrderParamsByQuery} from 'projects/trains/lib/urls/getTrainsOrderParamsByQuery/getTrainsOrderParamsByQuery';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import browserHistory from 'utilities/browserHistory/browserHistory';
import {cancelOrder} from 'projects/trains/lib/api/cancelOrder';

/**
 * Метод для кнопки "Выбрать места заново" в модалах с ошибками
 */
export default function choosePlacesAgain(
    directionAndIndex?: ITrainsDirectionAndIndex,
): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        if (!browserHistory) {
            return;
        }

        const {location} = browserHistory;
        const state = getState();
        const {
            trains: {
                order: {orderInfo},
            },
        } = state;
        const directionTrainsCount = directionTrainsCountSelector(state);

        dispatch(setOrderPageStatus(ORDER_PAGE_STATUSES.FETCHING_DATA));

        dispatch(
            batchActions([
                setOrderError(null),
                ...(directionAndIndex
                    ? getResetDirectionIndexActions(
                          directionAndIndex.direction,
                          directionAndIndex.index,
                      )
                    : flatMap(DIRECTIONS, direction => {
                          const count = directionTrainsCount[direction];

                          return flatMap(times(count), index =>
                              getResetDirectionIndexActions(direction, index),
                          );
                      })),
            ]),
        );

        const directionAndIndexDestination: ITrainsDirectionAndIndex =
            directionAndIndex ?? {
                direction: EDirection.FORWARD,
                index: 0,
            };

        dispatch(
            changeOrderStep({
                step: ORDER_STEP.PLACES,
                ...directionAndIndexDestination,
            }),
        );

        // Ждем отмены заказа, чтобы предыдущие выбранные места
        // были доступны к моменту нового фетчинга
        if (orderInfo) {
            dispatch(clearOrder());

            await cancelOrder(orderInfo);
        }

        await dispatch(
            requestAllTrainDetails(
                getTrainsOrderParamsByQuery(getQueryByLocation(location)),
            ),
        );

        dispatch(setOrderPageStatus(ORDER_PAGE_STATUSES.READY_TO_INTERACTION));
    };
}

function getResetDirectionIndexActions(
    direction: EDirection,
    index: number,
): Action[] {
    return [
        setPreferredGender({
            direction,
            index,
            data: null,
        }),
        setOrderCoachType({
            direction,
            index,
            data: null,
        }),
        setCoach({
            direction,
            index,
            data: null,
        }),
        setOrderCoachCategory({
            direction,
            index,
            data: null,
        }),
        clearPlaces({
            direction,
            index,
            data: undefined,
        }),
        clearRequirements({
            direction,
            index,
            data: undefined,
        }),
    ];
}
