import {batchActions} from 'redux-batched-actions';
import {Action} from 'typesafe-actions';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';

import {EDirection} from 'types/common/EDirection';
import {isTrainApiOrderError} from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/TTrainOrderError/TTrainApiOrderError';
import {ETrainOrderErrorCode} from 'projects/trains/components/TrainsOrderApp/components/BookError/types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {setOrderInfo} from 'reducers/trains/order/actions/data';
import {
    setOrderCreated,
    setOrderPageStatus,
} from 'reducers/trains/order/actions/flow';
import {setBookError, setOrderStep} from 'reducers/trains/order/actions/view';
import saveOffers from 'reducers/trains/order/thunk/saveOffers';
import saveTravelerAndPassengersToNotebook from 'reducers/trains/order/thunk/createOrder/utilities/saveTravelerAndPassengersToNotebook';

import passengersDataSelector from 'selectors/trains/order/passengersDataSelector';
import passengersForOrderSelector from 'selectors/trains/order/passengersForOrderSelector';

import {params} from 'utilities/metrika';
import {createNewOrder} from 'projects/trains/lib/api/createNewOrder';
import {logError} from 'utilities/logger/logError';
import {getTrainServices} from 'projects/trains/lib/order/getTrainServices';
import hasPassengersDataBonusCard from 'projects/trains/lib/order/fields/hasPassengersDataBonusCard';

export default function createOrder(): CustomThunkAction<Promise<boolean>> {
    return async (dispatch, getState): Promise<boolean> => {
        const state = getState();
        const {
            trains: {
                order: {contacts, traveler},
                userActions,
            },
        } = state;

        dispatch(setOrderPageStatus(ORDER_PAGE_STATUSES.RESERVING_TICKETS));

        try {
            const offers = await dispatch(saveOffers());
            const passengersData = passengersForOrderSelector(state);
            const withBonusCards = hasPassengersDataBonusCard(
                passengersDataSelector(state),
            );

            const orderInfo = await createNewOrder({
                userActions,
                offers,
                contacts,
                traveler,
                passengersData,
            });

            if (!contacts.useContacts || withBonusCards) {
                params({
                    trains: {
                        passengersStep: {
                            withAdditionalContacts: !contacts.useContacts,
                            withBonusCards,
                        },
                    },
                });
            }

            const error = getTrainServices(orderInfo).find(
                service => service.trainInfo.error,
            )?.trainInfo.error;

            if (error) {
                // TODO TRAINS: эмулируем ошибку, потому что бэкенд оркестратора не присылает статус 5xx
                throw {
                    response: {
                        data: error,
                    },
                };
            }

            saveTravelerAndPassengersToNotebook(
                contacts,
                passengersData[EDirection.FORWARD][0],
            );

            dispatch(
                batchActions([
                    setOrderInfo(orderInfo),
                    setOrderStep(ORDER_STEP.CONFIRM),
                    setOrderCreated(orderInfo.id),
                    setOrderPageStatus(
                        ORDER_PAGE_STATUSES.READY_TO_INTERACTION,
                    ),
                ]),
            );

            return true;
        } catch (error) {
            logError(
                {
                    message: `[YATRAVEL][TRAINS] Ошибка создания заказа`,
                    block: 'createOrder',
                },
                error,
            );
            console.error(error);

            const actions: Action[] = [
                setOrderPageStatus(ORDER_PAGE_STATUSES.READY_TO_INTERACTION),
            ];

            const errorData = error?.response?.data;

            actions.push(
                setBookError(
                    isTrainApiOrderError(errorData)
                        ? errorData
                        : {
                              code: ETrainOrderErrorCode.COMMON,
                          },
                ),
            );

            dispatch(batchActions(actions));
        }

        return false;
    };
}
