import moment from 'moment';

import {GENDER_TYPE} from 'projects/trains/constants/genders';
import {TRAIN_BONUS_CARDS} from 'projects/trains/constants/bonusCards';

import {
    ITrainOrderContacts,
    ITrainOrderField,
    ITrainPassenger,
    PassengerBonusCardsType,
} from 'reducers/trains/order/types';
import {EGender} from 'types/common/document/EGender';
import {ENotebookBonusCard} from 'server/api/NotebookApi/types/models';
import {INotebookCreatePassengerBonusCardParams} from 'server/api/NotebookApi/types/createPassengerBonusCard';
import {isNotNull} from 'types/utilities';
import {INotebookSavePassengerParams} from 'server/services/NotebookService/types/INotebookSaveTravelerAndPassengersService';

import {HUMAN_DATE_RU, ROBOT} from 'utilities/dateUtils/formats';
import {DOCUMENTS_TRAVELERS_KEYS} from 'projects/trains/lib/order/passengers/documentTypes';
import {logError} from 'utilities/logger/logError';
import getTitle from 'utilities/notebook/getTitle';

import {notebookBrowserProvider} from 'serviceProvider/notebook/notebookBrowserProvider';

// TODO: У пассажира в заказе должен быть gender c типом EGender
function getGender(gender: GENDER_TYPE): EGender {
    return gender === GENDER_TYPE.MALE ? EGender.MALE : EGender.FEMALE;
}

function getNotebookBonusCardType(
    bonusCardType: TRAIN_BONUS_CARDS,
): ENotebookBonusCard {
    switch (bonusCardType) {
        case TRAIN_BONUS_CARDS.BONUS_CARD: {
            return ENotebookBonusCard.RZD_BONUS;
        }

        case TRAIN_BONUS_CARDS.ROAD_CARD: {
            return ENotebookBonusCard.UNIVERSAL_ROAD;
        }
    }
}

function getBonusCards(
    bonusCards: PassengerBonusCardsType,
): INotebookCreatePassengerBonusCardParams[] {
    return (
        Object.entries(bonusCards) as [TRAIN_BONUS_CARDS, ITrainOrderField][]
    )
        .map(([type, bonusCardField]) => {
            if (!bonusCardField.value) {
                return null;
            }

            return {
                type: getNotebookBonusCardType(type),
                number: bonusCardField.value,
            };
        })
        .filter(isNotNull);
}

function preparePassenger(
    passenger: ITrainPassenger,
): INotebookSavePassengerParams {
    const gender = passenger.gender.value;
    const birthDate = passenger.birthDate.value;
    const documentType = passenger.passengerDocument.type.value;
    const documentNumber = passenger.passengerDocument.number.value;
    const citizenship = passenger.passengerDocument.country.value;
    const firstName = passenger.firstName.value;
    const lastName = passenger.lastName.value;
    const middleName = passenger.patronymic.value;

    // TODO: таких кейсов вообще не должно быть
    if (
        !gender ||
        !birthDate ||
        !documentType ||
        !documentNumber ||
        !citizenship ||
        !firstName ||
        !lastName
    ) {
        throw new Error('Неверные данные для сохранения в ЗКП');
    }

    return {
        title: getTitle(firstName, lastName, middleName),
        gender: getGender(gender),
        birthDate: moment(birthDate, HUMAN_DATE_RU).format(ROBOT),
        document: {
            type: DOCUMENTS_TRAVELERS_KEYS[documentType],
            number: documentNumber,
            citizenship,
            firstName,
            lastName,
            middleName: middleName || null,
        },
        bonusCards: getBonusCards(passenger.bonusCards),
    };
}

export default async function saveTravelerAndPassengersToNotebook(
    contacts: ITrainOrderContacts,
    passengers: ITrainPassenger[],
): Promise<void> {
    try {
        await notebookBrowserProvider.saveTravelerAndPassengers({
            traveler: {
                phone: contacts.phone.value || '',
                email: contacts.email.value || '',
                agree: true,
            },
            passengers: passengers.map(preparePassenger),
        });
    } catch (err) {
        logError(
            {
                message: '[YATRAVEL][TRAINS] Сохранение данных в ЗКП',
            },
            err,
        );
    }
}
