import moment from 'moment-timezone';
import {Moment} from 'moment';

import {Request} from '@yandex-data-ui/core/lib/types';
import {ETransport} from 'types/common/ETransport';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import {CustomThunkAction} from 'reducers/trains/customDispatch';

import {ROBOT, TIME} from 'utilities/dateUtils/formats';
import {delay} from 'utilities/async/delay';

import {trainsService} from 'serviceProvider';
import {rasp as raspProvider} from 'serviceProvider/rasp/rasp';

const DEFAULT_MAX_POLLS_COUNT = 3;
const DELAY_BETWEEN_POLLS_MS = 2000;

interface IParams {
    pointFrom: string;
    pointTo: string;
    when: Moment;
    number: string;
    maxPolls?: number;
}

/**
 * Экшен получает и возвращает сегмент из ручки тарифов сегментов.
 *
 * Необходимо, например, для получения сегмента для страницы заказа,
 * если сегмента нету в базе расписаний, т.е. это скорее всего рейс пополнения.
 */
export default function fetchTrainsOrderSegmentFromTariffs(
    {
        pointFrom,
        pointTo,
        when,
        number,
        maxPolls = DEFAULT_MAX_POLLS_COUNT,
    }: IParams,
    req?: Request,
): CustomThunkAction<Promise<ITrainsTariffApiSegment | null>> {
    return async (dispatch, getState) => {
        const {
            common: {
                features: {trainsAllowInternationalRoutes},
            },
        } = getState();

        const context = await raspProvider
            .provider(req?.container)
            .parseContext({
                from_key: pointFrom,
                to_key: pointTo,
                transport: ETransport.TRAIN,
            });

        if (!context.from || !context.from.timezone) {
            return null;
        }

        let isQuerying = true;
        let retriesCounter = 0;

        let segment = null;

        const date = when.format(ROBOT);
        const time = when.format(TIME);

        const startTime = moment.tz(when, context.from.timezone);

        const fetchParams = {
            when: date,
            pointFrom: pointFrom,
            pointTo: pointTo,
            startTime: startTime.format(),
            endTime: startTime.clone().add(1, 'minute').format(),
            allowInternationalRoutes: Boolean(trainsAllowInternationalRoutes),
        };

        while (isQuerying && retriesCounter++ < maxPolls) {
            const {segments, querying} = await trainsService
                .provider(req?.container)
                .getTariffs({
                    ...fetchParams,
                    poll: retriesCounter > 1 ? '1' : '0',
                });

            segment = segments.find(
                (tariffSegment: ITrainsTariffApiSegment) =>
                    number === tariffSegment.originalNumber &&
                    time ===
                        moment(tariffSegment.departure)
                            .tz(tariffSegment.stationFrom.timezone)
                            .format(TIME),
            );

            if (segment || !querying) {
                isQuerying = false;
            } else {
                await delay(DELAY_BETWEEN_POLLS_MS);
            }
        }

        return segment || null;
    };
}
