import {EOrderErrorType} from 'projects/trains/constants/orderErrors';
import {URLs} from 'constants/urls';

import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import {TrainsInsuranceStatus} from 'server/api/TrainsBookingApi/types/TrainsInsuranceStatus';
import EDeviceType from 'server/api/GenericOrderApi/types/common/EDeviceType';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {setOrderError} from 'reducers/trains/order/actions/view';
import {setPaymentCalled} from 'reducers/trains/order/actions/flow';

import {delay} from 'utilities/async/delay';
import {getFirstTrainService} from 'projects/trains/lib/complexOrder/getFirstTrainService';
import createLatestOrderInfoFetcher from 'projects/trains/lib/api/createLatestOrderInfoFetcher';
import getFinishPaymentUrl from 'utilities/url/getFinishPaymentUrl';

import {trainsBrowserProvider} from 'serviceProvider/trains/trainsBrowserProvider';
import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

export default function moveToPayment(): CustomThunkAction<void> {
    return async (dispatch, getState): Promise<void> => {
        try {
            const state = getState();
            const {
                trains: {order},
                common: {deviceType},
            } = state;
            const orderInfo = order.orderInfo;

            if (!orderInfo) {
                throw new Error('Отсутствует информация о заказе');
            }

            const {id: orderId} = orderInfo;

            const customerSource = deviceType.isDesktop
                ? EDeviceType.DESKTOP
                : EDeviceType.MOBILE;

            if (order.isPaymentCalled) {
                return;
            }

            if (
                orderInfo.payment?.paymentUrl &&
                orderInfo.state !== EGenericOrderState.PAYMENT_FAILED
            ) {
                return;
            }

            dispatch(setPaymentCalled(true));

            const trainService = getFirstTrainService(orderInfo);
            const withInsurance =
                order.insuranceIncluded &&
                trainService?.trainInfo.insuranceStatus ===
                    TrainsInsuranceStatus.PRICED;

            if (withInsurance) {
                await addInsurance(orderId);
            }

            const returnUrl = getFinishPaymentUrl(
                URLs.trainsOrderFinishPayment,
            );

            await genericOrderBrowserProvider.startPayment({
                orderId,
                source: customerSource,
                returnUrl,
            });
        } catch (e) {
            dispatch(
                setOrderError({
                    type: EOrderErrorType.INTERNAL_ERROR,
                }),
            );

            console.error(e);
        }
    };
}

async function addInsurance(orderId: string): Promise<void> {
    await trainsBrowserProvider.addInsurance({orderId});

    const getLatestOrderInfo = createLatestOrderInfoFetcher();
    let insuranceStatus: TrainsInsuranceStatus | undefined;

    do {
        await delay(1000);

        const order = await getLatestOrderInfo(orderId);

        insuranceStatus =
            getFirstTrainService(order)?.trainInfo.insuranceStatus;
    } while (
        !insuranceStatus ||
        ![
            TrainsInsuranceStatus.CHECKED_OUT,
            TrainsInsuranceStatus.CHECKOUT_FAILED,
        ].includes(insuranceStatus)
    );
}
