import {batchActions} from 'redux-batched-actions';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {EOrderErrorType} from 'projects/trains/constants/orderErrors';
import {isTrainsCoachType} from 'projects/trains/constants/coachType';

import {ITrainsOrderLocation} from 'types/trains/booking/ITrainsOrderLocation';
import {EDirection} from 'types/common/EDirection';

import {setOrderError} from 'reducers/trains/order/actions/view';
import {setCoachCategoryAndInitialCoach} from 'reducers/trains/order/thunk/setCoachCategoryAndInitialCoach';
import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    clearPlaces,
    clearRequirements,
    removePlacesFromOrder,
    setCoach,
    setExpandedServiceClass,
    setOrderCoachCategory,
    setOrderCoachType,
} from 'reducers/trains/order/actions/trains';

import experimentsSelector from 'selectors/common/experimentsSelector';

import getInitialCoach from 'projects/trains/lib/order/getInitialCoach';
import getClassFromServiceClassKey from 'projects/trains/lib/order/getClassFromServiceClassKey';
import {getCategories} from 'projects/trains/lib/order/coachesCategories';
import getCategoriesData from 'projects/trains/lib/order/getCategoriesData';
import {sortTariffClassKeys} from 'projects/trains/lib/segments/tariffs/sortTriffClassKeys';

import * as i18nBlock from 'i18n/trains-error-page-order';

export const prepareOrderPage =
    (
        query: ITrainsOrderLocation,
        direction: EDirection,
        index: number,
    ): CustomThunkAction<void> =>
    (dispatch, getState): void => {
        const {
            trains: {
                order,
                order: {orderStep},
            },
        } = getState();
        const state = getState();
        const train = order.trains[direction]?.[index];
        const trainDetails = train?.trainDetails?.trainDetails;
        const {trainsDefaultActiveTab} = experimentsSelector(state);

        if (!train || !trainDetails) {
            return;
        }

        const {coaches} = trainDetails;
        const {coachNumber, expandedServiceClassKey} = query;
        const schemeHasGroupedPlaces = Boolean(
            query.schemeHasGroupedPlaces?.[direction]?.[index],
        );

        const coach = coachNumber?.[direction]?.[index]
            ? getInitialCoach({
                  coaches,
                  query,
                  serviceClass: getClassFromServiceClassKey(
                      expandedServiceClassKey?.[direction]?.[index],
                  ),
                  direction,
                  index,
              })
            : null;
        let {coachType} = train;

        if (!coach || coach.type !== coachType) {
            // Если на странице подтверждения не оказалось вагона (например, выкупили последние места),
            // то сбрасываем состояние заказа.
            if (
                orderStep === ORDER_STEP.CONFIRM ||
                orderStep === ORDER_STEP.PAYMENT
            ) {
                dispatch(
                    batchActions([
                        setOrderCoachType({direction, index, data: null}),
                        setCoach({direction, index, data: null}),
                        setOrderCoachCategory({direction, index, data: null}),
                        setExpandedServiceClass({direction, index, data: null}),
                        clearPlaces({direction, index, data: undefined}),
                        clearRequirements({direction, index, data: undefined}),
                    ]),
                );

                return;
            }

            // Если на странице ввода данных не оказалось вагона,
            // то показываем ошибку
            if (orderStep === ORDER_STEP.PASSENGERS) {
                dispatch(
                    setOrderError({
                        type: EOrderErrorType.PLACES,
                        message:
                            i18nBlock.paymentDashErrorDashTimeoutDashContent(),
                    }),
                );

                return;
            }
        }

        if (!coachType && !trainsDefaultActiveTab) {
            const coachTypes = coaches.map(({type}) => type);

            coachType = sortTariffClassKeys(coachTypes, train.segment)[0];

            dispatch(setOrderCoachType({direction, index, data: coachType}));
        }

        const coachCategory = coach ? coach.category.value : null;

        // Вычищаем выбранные места из заказа если протух вагон или места уже заняты
        if (
            !coach ||
            train.orderPlaces.some(
                num => !coach.places.find(place => place.number === num),
            )
        ) {
            dispatch(
                removePlacesFromOrder({
                    direction,
                    index,
                    data: train.orderPlaces,
                }),
            );
        }

        if (
            coachCategory &&
            getCategories(coaches).find(({value}) => value === coachCategory)
        ) {
            dispatch(
                setCoachCategoryAndInitialCoach({
                    categoryOrType: coachCategory,
                    coach,
                    schemeHasGroupedPlaces,
                    direction,
                    index,
                }),
            );
        } else {
            const categoriesData = getCategoriesData(trainDetails);
            const firstCategory = categoriesData[0]?.category.value;
            const canSkipSetCoachType =
                isTrainsCoachType(firstCategory) && trainsDefaultActiveTab;

            if (trainsDefaultActiveTab && coachType && !coachCategory) {
                dispatch(
                    setCoachCategoryAndInitialCoach({
                        categoryOrType: coachType,
                        coach,
                        schemeHasGroupedPlaces,
                        isDefaultInitialization: true,
                        direction,
                        index,
                    }),
                );
            }

            if (firstCategory && !canSkipSetCoachType) {
                dispatch(
                    setCoachCategoryAndInitialCoach({
                        categoryOrType: firstCategory,
                        coach,
                        schemeHasGroupedPlaces,
                        isDefaultInitialization: true,
                        direction,
                        index,
                    }),
                );
            }
        }

        dispatch(clearRequirements({direction, index, data: undefined}));
    };
