import moment from 'moment';

import {ITrainsOrderLocation} from 'types/trains/booking/ITrainsOrderLocation';
import {EDirection} from 'types/common/EDirection';
import {ETrainDetailMock} from 'projects/testControlPanel/pages/TrainsPage/types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {prepareOrderPage} from 'reducers/trains/order/thunk/prepareOrderPage';
import {setTrainDetails} from 'reducers/trains/order/actions/trains';

import {trainsPurchaseIsAvailableSelector} from 'selectors/trains/trainsPurchaseIsAvailableSelector';
import trainsSelector from 'selectors/trains/order/trainsSelector';

import qsParseTimeFix from 'projects/trains/lib/qsParseTimeFix';
import {ROBOT_DATETIME} from 'utilities/dateUtils/formats';
import {logError} from 'utilities/logger/logError';
import {getMockImPath} from 'utilities/testUtils/getMockImPath';
import {isUnknownAxiosError} from 'utilities/error';
import {IPartnerError} from 'projects/trains/lib/errors/orderErrors';

import {trainsBrowserProvider} from 'serviceProvider/trains/trainsBrowserProvider';

export const requestTrainDetails =
    (
        query: ITrainsOrderLocation,
        direction: EDirection,
        index: number,
        fetchingState?: {
            needToStop: boolean;
        },
    ): CustomThunkAction<void> =>
    async (dispatch, getState): Promise<void> => {
        const state = getState();
        const purchaseIsAvailable = trainsPurchaseIsAvailableSelector(state);
        const trains = trainsSelector(state);

        if (!purchaseIsAvailable) {
            dispatch(
                setTrainDetails({
                    direction,
                    index,
                    data: {
                        errors: true,
                        trainDetails: null,
                        partnerError: null,
                    },
                }),
            );
        }

        const route = trains[direction]?.[index];
        const segment = route?.segment;
        const provider = route?.provider;

        if (!segment || !purchaseIsAvailable) {
            return;
        }

        const {stationFrom, stationTo, thread, departure} = segment;

        try {
            const number =
                ('originalNumber' in segment && segment.originalNumber) ||
                thread?.number;

            if (!number) {
                throw new Error('У сегмента нет номера');
            }

            const mockImPath = getMockImPath();
            const isAutoMock = mockImPath === ETrainDetailMock.AUTO;

            const trainDetails = await trainsBrowserProvider.getTrainDetails({
                stationFrom: stationFrom.id,
                stationTo: stationTo.id,
                when: moment
                    .tz(qsParseTimeFix(departure), stationFrom.railwayTimezone)
                    .format(ROBOT_DATETIME),
                number,
                provider,
                mockImAuto: isAutoMock,
                mockImPath: isAutoMock ? undefined : mockImPath,
            });

            if (fetchingState?.needToStop) return;

            dispatch(setTrainDetails({direction, index, data: trainDetails}));

            if (!trainDetails.errors) {
                dispatch(prepareOrderPage(query, direction, index));
            }
        } catch (err) {
            console.error(err);

            logError(
                {
                    message:
                        '[YATRAVEL][TRAINS] Не удалось получить trainDetails',
                    block: 'requestTrainDetails',
                },
                err,
            );

            if (isUnknownAxiosError<IPartnerError>(err))
                dispatch(
                    setTrainDetails({
                        direction,
                        index,
                        data: {
                            errors: true,
                            partnerError:
                                (err.response && err.response.data) ?? null,
                            trainDetails: null,
                        },
                    }),
                );
        }
    };
