import {batchActions} from 'redux-batched-actions';
import {Action} from 'typesafe-actions';

import {
    ITrainsOrderLocation,
    TDirectionIndexQueryValue,
} from 'types/trains/booking/ITrainsOrderLocation';
import {EDirection} from 'types/common/EDirection';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    addPlacesToOrder,
    setBeddingOption,
    setExpandedServiceClass,
    setOrderCoachType,
    setPreferredGender,
} from 'reducers/trains/order/actions/trains';

/**
 * Экшн диспатчит основную информацию о поезде. (места, выбранный гендер, расрытый класс вагонов и т.д.)
 *
 * @param query - query-параметры из урла
 * @param direction - направление поезда, для которого выставляются данные
 * @param index - индекс поезда по направлению, для которого выставляются данные
 */
export function setBaseTrainData(
    query: ITrainsOrderLocation,
    direction: EDirection,
    index: number,
): CustomThunkAction<void> {
    return (dispatch): void => {
        const place = getDirectionIndexValue(query.place, direction, index);
        const gender = getDirectionIndexValue(query.gender, direction, index);
        const bedding = getDirectionIndexValue(query.bedding, direction, index);
        const coachType = getDirectionIndexValue(
            query.coachType,
            direction,
            index,
        );
        const expandedServiceClassKey = getDirectionIndexValue(
            query.expandedServiceClassKey,
            direction,
            index,
        );

        const actions: Action[] = [];

        if (bedding !== undefined) {
            actions.push(setBeddingOption({direction, index, data: bedding}));
        }

        if (coachType !== undefined) {
            actions.push(
                setOrderCoachType({direction, index, data: coachType}),
            );
        }

        if (place !== undefined) {
            actions.push(addPlacesToOrder({direction, index, data: place}));
        }

        if (gender !== undefined) {
            actions.push(setPreferredGender({direction, index, data: gender}));
        }

        if (expandedServiceClassKey !== undefined) {
            actions.push(
                setExpandedServiceClass({
                    direction,
                    index,
                    data: expandedServiceClassKey,
                }),
            );
        }

        dispatch(batchActions(actions));
    };
}

function getDirectionIndexValue<T>(
    values: TDirectionIndexQueryValue<T> | undefined,
    direction: EDirection,
    index: number,
): T | undefined {
    return values?.[direction]?.[index];
}
