import {batchActions} from 'redux-batched-actions';
import {Action} from 'typesafe-actions';

import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';
import {
    isTrainsCoachType,
    TRAIN_COACH_TYPE,
} from 'projects/trains/constants/coachType';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {EDirection} from 'types/common/EDirection';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setCoach,
    setExpandedServiceClass,
    setInitialExpandedServiceClass,
    setOrderCoachCategory,
    setOrderCoachType,
    setSchemeHasGroupedPlaces,
} from 'reducers/trains/order/actions/trains';

import trainsOrderSelector from 'selectors/trains/order/trainsOrderSelector';
import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';

import getCoachClassKey from 'projects/trains/lib/order/getCoachClassKey';
import getCoachesByType from 'projects/trains/lib/order/getCoachesByType';
import getCoachesByCategory from 'projects/trains/lib/order/getCoachesByCategory';
import getCoachesWithEnoughPlaces from 'projects/trains/lib/order/coaches/getCoachesWithEnoughPlaces';

export const setCoachCategoryAndInitialCoach =
    ({
        categoryOrType,
        coach,
        schemeHasGroupedPlaces,
        isDefaultInitialization = false,
        direction,
        index,
    }: {
        categoryOrType: string | TRAIN_COACH_TYPE | null;
        coach: ITrainsCoach | null;
        schemeHasGroupedPlaces: boolean;
        isDefaultInitialization?: boolean;
        direction: EDirection;
        index: number;
    }): CustomThunkAction<void> =>
    (dispatch, getState): void => {
        const state = getState();
        const order = trainsOrderSelector(state);
        const passengers = passengersCountSelector(state);
        const train = order.trains[direction]?.[index];
        const trainDetails = train?.trainDetails?.trainDetails;
        const coachType = train?.coachType;
        const coachesWithPlacesForPassengers = getCoachesWithEnoughPlaces(
            trainDetails,
            passengers,
        );

        if (!trainDetails || !categoryOrType) {
            return;
        }

        const filteredCoaches = filterCoaches(
            coachesWithPlacesForPassengers,
            categoryOrType,
        );
        const initialCoach = coach || filteredCoaches[0];
        const serviceClassKey = getCoachClassKey({
            coach: initialCoach,
        });
        const actions: Action[] = [
            setOrderCoachCategory({direction, index, data: categoryOrType}),
        ];

        actions.push(setCoach({direction, index, data: initialCoach}));

        if (
            coach &&
            initialCoach.placeReservationType !==
                PLACE_RESERVATION_TYPE.USUAL &&
            schemeHasGroupedPlaces
        ) {
            actions.push(
                setSchemeHasGroupedPlaces({
                    direction,
                    index,
                    data: {
                        index: coach.index,
                        hasGroupedPlaces: schemeHasGroupedPlaces,
                    },
                }),
            );
        }

        actions.push(
            setExpandedServiceClass({
                direction,
                index,
                data: serviceClassKey,
            }),
        );

        if (isDefaultInitialization) {
            actions.push(
                setInitialExpandedServiceClass({
                    direction,
                    index,
                    data: serviceClassKey,
                }),
            );
        }

        if (isTrainsCoachType(categoryOrType) && categoryOrType !== coachType) {
            actions.push(
                setOrderCoachType({direction, index, data: categoryOrType}),
            );
        } else if (categoryOrType !== initialCoach.type) {
            actions.push(
                setOrderCoachType({direction, index, data: initialCoach.type}),
            );
        }

        dispatch(batchActions(actions));
    };

function filterCoaches(
    coaches: ITrainsCoach[],
    categoryOrType: string | TRAIN_COACH_TYPE,
): ITrainsCoach[] {
    if (isTrainsCoachType(categoryOrType)) {
        return getCoachesByType(coaches, categoryOrType);
    }

    return getCoachesByCategory(coaches, categoryOrType);
}
