import {batchActions} from 'redux-batched-actions';
import {Action} from 'typesafe-actions';
import _flattenDeep from 'lodash/flattenDeep';

import {TRAIN_BONUS_CARDS} from 'projects/trains/constants/bonusCards';
import {MAP_COMMON_DOCUMENT_TYPES_TO_TRAINS} from 'projects/trains/constants/documentTypes';

import {
    IPassenger,
    TSubmitPassenger,
    IFormContacts,
    ITrainsBookFormValues,
} from 'types/trains/booking/ITrainsBookFormValues';
import {ITrainCountrySelect} from '../types';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setLastName,
    setFirstName,
    setPatronymic,
    setGender,
    setBirthDate,
    setHasDiscountDocument,
    setEmailOrPhone,
} from 'reducers/trains/order/actions/singlePassengerData';
import {
    setDocumentType,
    setDocumentNumber,
    setDocumentCountry,
    setDocumentValidDate,
} from 'reducers/trains/order/actions/passengerDocument';
import {setBonusCard} from 'reducers/trains/order/actions/bonusCards';

import {mapSexToTrainsGender} from 'projects/trains/components/TrainsOrderPage/PassengersStep/utilities/mapPassengerDataToForm';

const DEFAULT_PATRONYMIC = '-';

function isValidDocument(
    formPassenger: IPassenger,
): formPassenger is TSubmitPassenger {
    return Boolean(
        formPassenger.lastName &&
            formPassenger.firstName &&
            formPassenger.sex &&
            formPassenger.birthdate &&
            formPassenger.documentType &&
            formPassenger.documentNumber &&
            formPassenger.citizenship,
    );
}

export function mapPassengerDataToActions(
    formPassenger: IPassenger,
    countries: ITrainCountrySelect[],
    index: number,
    constacts?: IFormContacts,
): Action[] {
    if (!isValidDocument(formPassenger)) {
        return [];
    }

    const {
        lastName,
        firstName,
        patronymicName,
        sex,
        birthdate,
        documentType,
        documentNumber,
        citizenship,
        documentValidDate,
        bonusCards,
        hasDiscountDocument,
        additionalContact = '',
    } = formPassenger;
    const gender = mapSexToTrainsGender(sex);
    const formCountry = countries.find(x => x.code2 === citizenship);
    const useOnlyMainContacts = constacts?.useOnlyMainContacts;

    if (!gender || !formCountry) {
        return [];
    }

    const bonusCardActions: Action[] = Object.values(TRAIN_BONUS_CARDS).map(
        bonusCardType =>
            setBonusCard(
                {
                    type: bonusCardType,
                    number: bonusCards?.[bonusCardType],
                },
                index,
            ),
    );

    const hasDiscountDocumentActions = hasDiscountDocument
        ? [setHasDiscountDocument(true, index)]
        : [];

    return [
        setLastName(lastName, index),
        setFirstName(firstName, index),
        setPatronymic(patronymicName?.trim() || DEFAULT_PATRONYMIC, index),
        setGender(gender, index),
        setBirthDate(birthdate, index),
        setDocumentType(
            MAP_COMMON_DOCUMENT_TYPES_TO_TRAINS[documentType],
            index,
        ),
        setDocumentNumber(documentNumber, index),
        setDocumentCountry(formCountry.geoId, index),
        setDocumentValidDate(documentValidDate || '', index),
        setEmailOrPhone(useOnlyMainContacts ? '' : additionalContact, index),
        ...hasDiscountDocumentActions,
        ...bonusCardActions,
    ];
}

export default function setOrderPassengers(
    formValues?: ITrainsBookFormValues,
): CustomThunkAction<void> {
    return async function (dispatch, getState): Promise<void> {
        if (!formValues || !formValues.passengers) {
            return;
        }

        const {passengers, contacts} = formValues;
        const countries = getState().trains.order.availableCountries;
        const {adults = [], children = [], babies = []} = passengers;

        const adultActions = adults.map((adultPassenger, index) =>
            mapPassengerDataToActions(
                adultPassenger,
                countries,
                index,
                contacts,
            ),
        );
        const childrenActions = children.map((childrenPassenger, index) =>
            mapPassengerDataToActions(
                childrenPassenger,
                countries,
                index + (adults?.length || 0),
                contacts,
            ),
        );
        const babiesActions = babies.map((babiesPassenger, index) =>
            mapPassengerDataToActions(
                babiesPassenger,
                countries,
                index + (adults?.length || 0) + (children?.length || 0),
                contacts,
            ),
        );

        dispatch(
            batchActions(
                _flattenDeep([adultActions, childrenActions, babiesActions]),
            ),
        );
    };
}
