import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ITrainRoutePart} from 'types/trains/booking/ITrainsOrderLocation';
import {EDirection} from 'types/common/EDirection';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import fetchTrainsOrderSegmentFromTariffs from 'reducers/trains/order/thunk/fetchTrainsOrderSegmentFromTariffs';

import {ROBOT} from 'utilities/dateUtils/formats';

import {raspBrowserProvider as raspService} from 'serviceProvider/rasp/raspBrowserProvider';

const MAX_FETCH_SEGMENT_POLLS_FROM_TARIFFS = 6;

export default function trainsFetchOrderSegment(
    partRoute: ITrainRoutePart,
    direction: EDirection,
    index: number,
): CustomThunkAction<Promise<TTrainsStoreOrderSegment | null>> {
    return async function (
        dispatch,
        getState,
    ): Promise<TTrainsStoreOrderSegment | null> {
        const {
            trains: {context, order},
        } = getState();

        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        // Уже получили сегмент
        const existedSegment = order.trains[direction][index]?.segment;

        if (existedSegment) {
            return existedSegment;
        }

        const {when, number, fromId, toId} = partRoute;

        if (!when || !number) {
            return null;
        }

        const date = when.format(ROBOT);
        const time = when.format('HH.mm');

        try {
            return raspService.getTrainOrderSegment({
                pointFrom: fromId,
                pointTo: toId,
                number,
                departure: `${date}T${time}`,
            });
        } catch (e) {
            try {
                return await dispatch(
                    fetchTrainsOrderSegmentFromTariffs({
                        when,
                        pointFrom: fromId,
                        pointTo: toId,
                        number,
                        maxPolls: MAX_FETCH_SEGMENT_POLLS_FROM_TARIFFS,
                    }),
                );
            } catch (err) {
                return null;
            }
        }
    };
}
