import flatMap from 'lodash/flatMap';
import {batchActions} from 'redux-batched-actions';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {EOrderErrorType} from 'projects/trains/constants/orderErrors';
import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';

import {ITrainsOrderLocation} from 'types/trains/booking/ITrainsOrderLocation';
import {DIRECTIONS} from 'types/common/EDirection';
import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import {EPubSubEvent} from 'types/EPubSubEvent';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {setBaseTrainData} from 'reducers/trains/order/thunk/setBaseTrainData';
import {requestTrainDetails} from 'reducers/trains/order/thunk/requestTrainDetails';
import {
    setDirection,
    setIndex,
    setOrderError,
    setOrderStep,
} from 'reducers/trains/order/actions/view';
import {setOrderContext} from 'reducers/trains/order/thunk/setOrderContext';
import {
    cancelOrder,
    setOrderPageStatus,
    startOrder,
} from 'reducers/trains/order/actions/flow';
import {requestOrderInfo} from 'reducers/trains/order/thunk/requestOrderInfo';
import {updatePurchaseAvailability} from 'reducers/trains/actions/updatePurchaseAvailability';
import trainsFetchOrderSegment from 'reducers/trains/order/thunk/trainsFetchOrderSegment';
import {
    setOrderSegment,
    setProvider,
    setTrainDetails,
} from 'reducers/trains/order/actions/trains';
import validateTrainsPassengersCount from 'reducers/trains/order/thunk/validateTrainsPassengersCount';
import restoreTrainsPassengersCount from 'reducers/trains/order/thunk/restorePassengersCount';
import {setOrderPassengersDataFromQuery} from 'reducers/trains/order/thunk/setOrderPassengersDataFromQuery';
import {TOrderStepDescription} from 'reducers/trains/order/thunk/changeOrderStep';
import {resetOrder} from 'reducers/trains/order/actions/common';

import {trainsPurchaseIsAvailableSelector} from 'selectors/trains/trainsPurchaseIsAvailableSelector';

import {isOrderStepAfterReserve} from 'projects/trains/lib/order/orderStatus';
import {subscribeOnce} from 'utilities/pubSub';

export default function trainsInitOrderApp(
    orderStepDescription: TOrderStepDescription,
    query: ITrainsOrderLocation,
): CustomThunkAction<void> {
    return async function (dispatch, getState): Promise<void> {
        const fetchingState = {
            needToStop: false,
        };

        subscribeOnce(EPubSubEvent.TRAINS_LEAVE_BOOK_PAGES, () => {
            fetchingState.needToStop = true;

            dispatch(batchActions([cancelOrder(), resetOrder()]));
        });

        if (orderStepDescription.step === ORDER_STEP.PLACES) {
            dispatch(setDirection(orderStepDescription.direction));
            dispatch(setIndex(orderStepDescription.index));
        }

        dispatch(setOrderPageStatus(ORDER_PAGE_STATUSES.FETCHING_DATA));
        dispatch(setOrderStep(orderStepDescription.step));

        dispatch(startOrder());

        await dispatch(setOrderContext(query));
        await dispatch(updatePurchaseAvailability());

        dispatch(setOrderPassengersDataFromQuery(query));

        const promises = flatMap(DIRECTIONS, direction => {
            const routes = query[direction] ?? [];

            return routes.map(async (route, index) => {
                dispatch(setBaseTrainData(query, direction, index));

                dispatch(
                    setProvider({
                        direction,
                        index,
                        data: route.provider || '',
                    }),
                );

                let orderSegment: TTrainsStoreOrderSegment | null = null;

                try {
                    orderSegment = await dispatch(
                        trainsFetchOrderSegment(route, direction, index),
                    );
                } catch (err) {
                    console.log('Ошибка получения сегмента', err);
                }

                /*
                 * Если получить сегмент не получилось - отображаем ошибку, т.к.
                 * без сегмента мы не можем понять какой поезд нужен пользователю.
                 */
                if (!orderSegment) {
                    dispatch(
                        setTrainDetails({
                            direction,
                            index,
                            data: {
                                trainDetails: null,
                                errors: true,
                                partnerError: null,
                            },
                        }),
                    );

                    return;
                }

                dispatch(
                    setOrderSegment({direction, index, data: orderSegment}),
                );

                await dispatch(
                    requestTrainDetails(query, direction, index, fetchingState),
                );
            });
        });

        await Promise.all(promises);

        if (fetchingState.needToStop) return;

        dispatch(restoreTrainsPassengersCount());
        dispatch(validateTrainsPassengersCount());

        const purchaseIsAvailable = trainsPurchaseIsAvailableSelector(
            getState(),
        );
        const isReserved = isOrderStepAfterReserve(orderStepDescription.step);

        if (isReserved && query.id) {
            await dispatch(requestOrderInfo(query.id));
        }

        if (!purchaseIsAvailable && isReserved) {
            dispatch(setOrderError({type: EOrderErrorType.COMMON}));
        }

        dispatch(setOrderPageStatus(ORDER_PAGE_STATUSES.READY_TO_INTERACTION));
    };
}
