/* eslint-disable camelcase */
import {Moment} from 'moment-timezone';
import {Overwrite} from 'utility-types';

import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {GENDER_TYPE} from 'projects/trains/constants/genders';
import {TRAINS_DOCUMENT_TYPES} from 'projects/trains/constants/documentTypes';
import {TRAIN_BONUS_CARDS} from 'projects/trains/constants/bonusCards';
import {
    ADDITIONAL_REQUIREMENTS,
    ARRANGEMENT_REQUIREMENTS,
} from 'projects/trains/constants/requirements';
import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';
import {EOrderErrorType} from 'projects/trains/constants/orderErrors';

import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import {
    ITrainsCoachApi,
    ITrainsDetailsApi,
    ITrainsDetailsInfoServiceResponse,
} from 'server/services/TrainsService/types/ITrainsDetailsInfoServiceResponse';
import ITrainsCreateOrderLoyaltyCard from 'server/api/TrainsBookingApi/types/ITrainsCreateOrderLoyaltyCard';
import {TrainBookedTariffCode} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoTicket';
import {TrainsOrderRefundType} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoResponse';
import {ETrainsOrderInfoWarningCode} from 'server/api/TrainsBookingApi/types/ETrainsOrderInfoWarningCode';
import {IPassengerWithDocumentsAndBonusCardsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {EGender} from 'types/common/document/EGender';
import {TrainsOrderStatus} from 'server/api/TrainsBookingApi/types/TrainsOrderStatus';
import {TrainsInsuranceStatus} from 'server/api/TrainsBookingApi/types/TrainsInsuranceStatus';
import EOriginalPaymentErrorCode from 'types/common/EOriginalPaymentErrorCode';
import {EPlacesView} from 'types/trains/booking/EPlacesView';
import {EDirection, isEDirection} from 'types/common/EDirection';
import ITrainsOrderInfoStation from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoStation';
import IContactInfo from 'server/api/GenericOrderApi/types/common/IContactInfo';
import {ITrainGenericService} from 'server/api/GenericOrderApi/types/common/TGenericService';
import {TrainsOrderInfoTicketRzhdStatus} from 'server/api/TrainsBookingApi/types/TrainsOrderInfoTicketRzhdStatus';

import {TProviderState} from 'reducers/trains/order/reducers/provider';
import {TNonRefundableTariffState} from 'reducers/trains/order/reducers/nonRefundableTariff';

import {EUFSErrorCodes} from 'projects/trains/lib/errors/orderErrors';

export type TrainsPassengersCount = Record<PASSENGERS_TYPES, number>;

export interface ITrainOrderField<TValue = string> {
    value?: TValue;
}

export interface ITrainOrderDocument {
    type: ITrainOrderField<TRAINS_DOCUMENT_TYPES | null>;
    number: ITrainOrderField;
    country: ITrainOrderField<number | null>;
    validDate: ITrainOrderField;
}

export type PassengerBonusCardsType = Partial<
    Record<TRAIN_BONUS_CARDS, ITrainOrderField>
>;

export interface ITrainPassenger {
    lastName: ITrainOrderField;
    firstName: ITrainOrderField;
    patronymic: ITrainOrderField;
    birthDate: ITrainOrderField;
    gender: ITrainOrderField<GENDER_TYPE | null>;
    emailOrPhone: ITrainOrderField;
    hasDiscountDocument: boolean;
    passengerDocument: ITrainOrderDocument;
    bonusCards: PassengerBonusCardsType;
    ageGroup: PASSENGERS_TYPES;
    isNonRefundableTariff: boolean;
}

export interface ITrainPassengerContacts {
    phone?: string;
    email?: string;
}

export interface ITrainOrderContacts {
    email: ITrainOrderField;
    phone: ITrainOrderField;
    useContacts: ITrainOrderField<boolean>;
}

interface ISittingCountRequirements {
    nearWindow: number;
    nearPassage: number;
}

export interface ICompartmentCountRequirements {
    upper: number;
    bottom: number;
}

export type CountRequirementsType =
    | ISittingCountRequirements
    | ICompartmentCountRequirements;

export type StoreyRequirementsType = number | 'irrelevant';

export interface ITrainsRequirements {
    count?: CountRequirementsType;
    storey?: StoreyRequirementsType;
    arrangement?: ARRANGEMENT_REQUIREMENTS;
    additional?: ADDITIONAL_REQUIREMENTS;
}

export interface IOrderError {
    type: EOrderErrorType;
    title?: string;
    message?: React.ReactNode;
    description?: string;
    params?: string[];
    data?: {
        code?: EUFSErrorCodes;
    };
}

export interface ITrainsCoach extends ITrainsCoachApi {}

export interface ITrainsDetails
    extends Overwrite<
        ITrainsDetailsApi,
        {
            coaches: ITrainsCoach[];
        }
    > {}

export interface ITrainDetailsInfo
    extends Overwrite<
        ITrainsDetailsInfoServiceResponse,
        {
            trainDetails: ITrainsDetails | null;
        }
    > {}

interface IOrderTicketPayment {
    amount: number;
    beddingAmount: number;
    fee: number;
}

export interface IOrderTicketTariffInfo {
    minAge: number;
    maxAge: number;
    minAgeIncludesBirthday: boolean;
    maxAgeIncludesBirthday: boolean;
    title: string;
    code: string;
    needDocument: boolean;
    withoutPlace: boolean;
}

interface IOrderTicketRefund {
    amount: number;
    type?: TrainsOrderRefundType;
}

export interface IOrderTicket {
    blankId: string;
    rzhdStatus: TrainsOrderInfoTicketRzhdStatus | null;
    places: string[];
    placesType: {
        code: string;
        description: string;
    };
    tariffInfo: IOrderTicketTariffInfo | null;
    rawTariffTitle: string;
    bookedTariffCode: TrainBookedTariffCode | null;
    refund: IOrderTicketRefund | null;
    pending: boolean;
    isRefunding: boolean;
    amount: number;
    payment: IOrderTicketPayment;
    /**
     * https://st.yandex-team.ru/TRAVELFRONT-3629
     */
    canChangeElectronicRegistrationTill: string | null;
    canReturnTill: string | null;
}

interface IOrderInsuranceCompensationVariant {
    compensation: number;
    event: 'Death' | 'Injury' | 'Disability'; // TODO TRAINS: есть ли какие-нибудь еще варианты?
}

export interface IOrderPassengerInsurance {
    ordered: boolean;
    refunded: boolean;
    compensation: number;
    amount: number;
    compensationVariants: IOrderInsuranceCompensationVariant[];
    operationId: null; // TODO TRAINS: добавить тип помимо null
}

export interface IOrderPassenger {
    tickets: IOrderTicket[];
    customerId: string;
    docId: string;
    citizenship: string;
    firstName: string;
    lastName: string;
    patronymic: string;
    birthDate: string;
    age: number;
    docType: TRAINS_DOCUMENT_TYPES;
    sex: 'M' | 'F';
    insurance: IOrderPassengerInsurance;
    isNonRefundableTariff: boolean;
}

export enum RefundPaymentStatus {
    NEW = 'new',
    FAILED = 'failed',
    UNKNOWN = 'unknown',
    DONE = 'done',
    TRY_LATER = 'try_later',
    CREATED = 'created',
}

export interface IRefundPayment {
    blankIds: string[]; // Если был возврат страховки, то blankIds будет пустым
    paymentRefundReceiptUrl: string;
    refundCreatedAt: string;
    insuranceIds: string[];
    refundPaymentStatus: RefundPaymentStatus;
}

export interface IOrderWarning {
    code: ETrainsOrderInfoWarningCode;
    from?: string;
    to?: string;
}

// You can drop ITrainRailwayStation after support ITrainsSettlement in API

export interface ITrainsHpGenericOrder {
    id: string;
    prettyId: string | null;
    status: TrainsOrderStatus;
    customerInfo: IContactInfo;
    services: ITrainGenericService[];
}

export interface ITrainsOrderDetails {
    uid: string; // UID заказа в базе Поездов.
    orderNumber: string | null; // Номер заказа. Появляется после оплаты.
    partnerOrderNumber?: string; // Номер заказа у партнераы
    status: TrainsOrderStatus;
    passengers: IOrderPassenger[];
    carType: TRAIN_COACH_TYPE;
    trainName: string | null;
    carNumber: string;
    trainNumber: string;
    trainTitle?: string;
    brandTitle?: string;
    trainTicketNumber: string;
    coachOwner: string;
    isSuburban: boolean;
    twoStorey: boolean;
    startStation: string;
    endStation: string;
    stationFrom: ITrainsOrderInfoStation;
    stationTo: ITrainsOrderInfoStation;
    departure: string;
    arrival: string;
    reservedTo: string;
    ticketsStatusFreezesAt: string | null;
    rebookingAvailable: boolean;
    compartmentGender: GENDER_TYPE; // TODO TRAINS: в оркестраторе этот тип может быть null. После перехода на оркестратор нужно использовать null значения.
    insuranceStatus: TrainsInsuranceStatus;
    /**
     * im
     */
    partner: string;
    warnings: IOrderWarning[];
    error: IOrderError | null;
    specialNotice: string | null;
    isOnlyFullReturnPossible: boolean;
    paymentUrl: string | null;
    paymentCode: EOriginalPaymentErrorCode | null;
    paymentReceiptUrl: string | null;
    refundPayments: IRefundPayment[];
    userInfo: IContactInfo;
    isFullRefundPossible: boolean;
    services?: ITrainGenericService[];
    companyTitle?: string;
}

export interface ITrainsCoachTariff {
    /** Код тарифа */
    code: TrainBookedTariffCode;
    /** Верхняя граница возраста */
    maxAge: number;
    /** Строгость сравнения */
    maxAgeIncludesBirthday: boolean;
    /** Нижняя граница возраста */
    minAge: number;
    /** Строгость сравнения */
    minAgeIncludesBirthday: boolean;
    /** Необходимо ли предъявить документ, чтобы подтвердить тариф */
    needDocument: boolean;
    /** Название тарифа */
    title: string;
    /** Название тарифа от партнера */
    rawTitle?: string;
    /** Тариф без места */
    withoutPlace: boolean;
}

export interface IReservationVariant {
    passengers: {
        baby: number;
        child: number;
        full: number;
    };
    amount: number;
    nonRefundableAmount: number | null;
    placesCount: number;
    giveChildWithoutPlace: boolean;
}

export interface IMappedCoach extends ITrainsCoach {
    throughArrivalMoment?: Moment;
}

export interface IStoreTrainTraveler {
    accountPassengers: IPassengerWithDocumentsAndBonusCardsDTO[];
    passengers: IStoreTrainTravelerPassenger[];
}

// TODO наследовать от IPassengerDTO
export interface IStoreTrainTravelerPassenger {
    itn: Nullable<string>;
    documents: IStoreTravelerPassengerDocument[];
    train_notifications_enabled: boolean;
    title: string;
    gender: ITrainOrderField<EGender>;
    created_at: string;
    updated_at: string;
    id: string;
    phone_additional: Nullable<string>;
    phone: Nullable<string>;
    email: Nullable<string>;
    loyaltyCards: ITrainsCreateOrderLoyaltyCard[];
    birthDate: ITrainOrderField;
}

export interface IStoreTravelerPassengerDocument extends ITrainOrderDocument {
    id: string;
    passengerID: string;
    firstName: ITrainOrderField;
    patronymic: ITrainOrderField;
    lastName: ITrainOrderField;
    clicked: boolean;
}

export interface IAdditionalSchemeInfo {
    index: number; // порядковый номер (coach.index)
    hasGroupedPlaces: boolean; // схема поддерживает выбор купе целиком
}

export interface ITrainCountry {
    id: number;
    title: string;
    geoId: number;
    code2: string;
    code3: string;
}

export interface ITrainCountrySelect extends ITrainCountry {
    value: number; // == geoId
    data: string; // == title
}

export interface ITrainOrderState {
    segment: TTrainsStoreOrderSegment | null;
    coach: ITrainsCoach | null;
    gender: GENDER_TYPE | null;
    coachCategory: string | null;
    coachType: TRAIN_COACH_TYPE | null;
    orderPlaces: number[];
    placesViewType: EPlacesView;
    trainDetails: ITrainDetailsInfo | null;
    requirements: ITrainsRequirements;
    beddingOption: boolean;
    additionalSchemeInfo: IAdditionalSchemeInfo[];
    expandedServiceClass: string | null;
    initialExpandedServiceClass: string | null;
    provider: TProviderState;
    nonRefundableTariff: TNonRefundableTariffState;
}

export type TTrainsOrderState = Record<
    EDirection,
    Record<number, ITrainOrderState>
>;

export interface ITrainDirectionIndexPayload<Payload = undefined> {
    direction: EDirection;
    index: number;
    data: Payload;
}

export function isTrainDirectionIndexPayload<P>(
    candidate: unknown,
): candidate is ITrainDirectionIndexPayload<P> {
    if (typeof candidate !== 'object' || !candidate) {
        return false;
    }

    const payload = candidate as Record<string, unknown>;

    return isEDirection(payload.direction) && typeof payload.index === 'number';
}
