import {batchActions} from 'redux-batched-actions';

import {isFilledContextPoint} from 'reducers/trains/context/types';
import {isNotNull} from 'types/utilities';
import {IFilledContextPoint} from 'server/api/RaspApi/types/IRaspParseContextApiResponse';
import {
    ETrainsSuggestType,
    ITrainsSuggest,
} from 'types/trains/common/ITrainsApiSuggest';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setTrainsSearchFormFromFieldAction,
    setTrainsSearchFormStartDateFieldAction,
    setTrainsSearchFormToFieldAction,
    setTrainsSearchFormEndDateFieldAction,
    setTrainsSearchFormIsRoundTripFieldAction,
} from 'reducers/trains/searchForm/actions';

import {isCity} from 'utilities/geo/geoObjectType';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

/**
 * Преобразует данные точки отправления/прибытия из поискового контекста к виду поисковой формы
 *
 * @param point - пункт прибытия/отправления
 */
const getPointData = (point: IFilledContextPoint): ITrainsSuggest => {
    return {
        slug: point.slug,
        title: point.title,
        region: point.region.title,
        settlement: point.settlement.title,
        country: point.country.title,
        popularTitle: point.popularTitle || '',
        fullTitle: point.popularTitle || '',
        stationType: '',
        pointKey: point.key,
        objType: isCity(point.key)
            ? ETrainsSuggestType.SETTLEMENT
            : ETrainsSuggestType.STATION,
        objId: point.key,
        isUniqueTitle: true,
    };
};

/**
 * Заполнение поисковой формы из поискового контекста
 */
export function fillFormFromContextThunkAction(): CustomThunkAction<void> {
    return (dispatch, getState): void => {
        const {
            trains: {context},
        } = getState();

        const {from, to, originalWhen, returnWhen} = context;

        dispatch(
            batchActions(
                [
                    from && isFilledContextPoint(from)
                        ? setTrainsSearchFormFromFieldAction({
                              inputValue: from.title,
                              source: ESuggestSource.SEARCH_CONTEXT,
                              selectedValue: getPointData(from),
                          })
                        : null,
                    to && isFilledContextPoint(to)
                        ? setTrainsSearchFormToFieldAction({
                              inputValue: to.title,
                              source: ESuggestSource.SEARCH_CONTEXT,
                              selectedValue: getPointData(to),
                          })
                        : null,
                    originalWhen
                        ? setTrainsSearchFormStartDateFieldAction(originalWhen)
                        : null,
                    returnWhen
                        ? setTrainsSearchFormEndDateFieldAction(returnWhen)
                        : null,
                    returnWhen
                        ? setTrainsSearchFormIsRoundTripFieldAction(true)
                        : null,
                ].filter(isNotNull),
            ),
        );
    };
}
