import {batchActions} from 'redux-batched-actions';

import {IPointInfo} from 'server/api/TransformSearchApi/types/IPointInfo';
import {
    ETrainsSuggestType,
    ITrainsSuggest,
} from 'types/trains/common/ITrainsApiSuggest';
import {ITransformSearchContextRequestParams} from 'server/api/TransformSearchApi/types/ITransformSearchContextRequestParams';
import {ITrainsSearchFormPointField} from 'projects/trains/components/SearchForm/types';
import {ITransformSearchContextResponse} from 'server/api/TransformSearchApi/types/ITransformSearchContextResponse';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    setTrainsSearchFormEndDateFieldAction,
    setTrainsSearchFormFromFieldAction,
    setTrainsSearchFormIsRoundTripFieldAction,
    setTrainsSearchFormStartDateFieldAction,
    setTrainsSearchFormToFieldAction,
} from 'reducers/trains/searchForm/actions';

import {
    ESuggestSource,
    ISuggestValue,
} from 'components/SearchSuggest/SearchSuggest';

import {crossSearchProvider} from 'serviceProvider/common/crossSearch/crossSearchProvider';

const getTrainsSuggestFromPointInfo = (
    pointInfo: IPointInfo,
): ISuggestValue<ITrainsSuggest> => {
    return {
        inputValue: pointInfo.title,
        source: ESuggestSource.CROSS_SEARCH,
        selectedValue: {
            objId: pointInfo.pointKey ?? '',
            objType: ETrainsSuggestType.SETTLEMENT,
            fullTitle: pointInfo.title,
            title: pointInfo.title,
            slug: pointInfo.slug ?? '',
            settlement: '',
            region: pointInfo.regionTitle ?? '',
            country: pointInfo.countryTitle,
            popularTitle: pointInfo.title,
            stationType: '',
            pointKey: pointInfo.pointKey ?? '',
            isUniqueTitle: true,
        },
    };
};

export const transformSearchContext = (
    params: Omit<ITransformSearchContextRequestParams, 'vertical'>,
): CustomThunkAction<Promise<ITransformSearchContextResponse>> => {
    return async (dispatch): Promise<ITransformSearchContextResponse> => {
        dispatch(
            batchActions([
                setTrainsSearchFormFromFieldAction({
                    selectedValue: false,
                    inputValue: '',
                    source: ESuggestSource.CROSS_SEARCH,
                }),
                setTrainsSearchFormToFieldAction({
                    selectedValue: false,
                    inputValue: '',
                    source: ESuggestSource.CROSS_SEARCH,
                }),
                setTrainsSearchFormStartDateFieldAction(null),
                setTrainsSearchFormEndDateFieldAction(null),
                setTrainsSearchFormIsRoundTripFieldAction(false),
            ]),
        );

        const crossSearchInfo = await crossSearchProvider
            .provider()
            .transformSearchContext({
                ...params,
                vertical: 'trains',
            });

        const {from, to, dateForward, dateBackward} = crossSearchInfo;

        const fromField: ITrainsSearchFormPointField = from
            ? getTrainsSuggestFromPointInfo(from)
            : {
                  selectedValue: false,
                  inputValue: '',
                  source: ESuggestSource.INPUT,
              };
        const toField: ITrainsSearchFormPointField = to
            ? getTrainsSuggestFromPointInfo(to)
            : {
                  selectedValue: false,
                  inputValue: '',
                  source: ESuggestSource.INPUT,
              };

        dispatch(
            batchActions([
                setTrainsSearchFormFromFieldAction(fromField),
                setTrainsSearchFormToFieldAction(toField),
                setTrainsSearchFormStartDateFieldAction(dateForward ?? null),
                ...(dateBackward
                    ? [
                          setTrainsSearchFormEndDateFieldAction(
                              dateBackward ?? null,
                          ),
                          setTrainsSearchFormIsRoundTripFieldAction(true),
                      ]
                    : []),
            ]),
        );

        return crossSearchInfo;
    };
};
