import * as localStorage from 'local-storage';

import IGetTripRequestRequest from 'server/api/TripsApi/types/IGetTripRequestRequest';

import {CustomThunkAction} from 'reducers/trains/customDispatch';
import {
    getTrip,
    hideHotelCrossSaleBlock,
    restoreHiddenHotelsCrossSaleBlocks,
} from 'reducers/trips/tripPage/actions';

import {isUnknownAxiosError} from 'utilities/error';
import getNotLoadedBlockTypes from 'projects/account/pages/TripPage/utilities/getNotLoadedBlockTypes';
import patchBlocksInTrip from 'projects/account/pages/TripPage/utilities/patchBlocksInTrip';
import {logError} from 'utilities/logger/logError';

import {browserTripsProvider} from 'serviceProvider/trips/browserTripsProvider';

const HIDDEN_HOTEL_CROSS_SALE_BLOCKS_LS_KEY =
    'tripPage/hiddenHotelsCrossSaleBlocks';

export function fetchTrip(
    params: IGetTripRequestRequest,
): CustomThunkAction<Promise<void>> {
    return async (dispatch): Promise<void> => {
        try {
            dispatch(getTrip.request(params));

            const {trip} = await browserTripsProvider.getTrip(params);

            dispatch(getTrip.success(trip));

            const notLoadedBlockTypes = getNotLoadedBlockTypes(trip);

            if (!notLoadedBlockTypes.length) {
                return;
            }

            try {
                const {blocks} = await browserTripsProvider.getTripAsyncBlock({
                    tripId: params.tripId,
                    blockTypes: notLoadedBlockTypes,
                });

                const patchedTrip = patchBlocksInTrip(trip, blocks);

                dispatch(getTrip.success(patchedTrip));
            } catch (e) {
                logError(
                    {
                        message: `[YATRAVEL][TRIPS] getTripAsyncBlock failed `,
                    },
                    e,
                );
            }
        } catch (error) {
            const status = isUnknownAxiosError(error)
                ? error.response?.status
                : undefined;

            dispatch(getTrip.failure(status));
        }
    };
}

export function hideHotelCrossSale(tripId: string): CustomThunkAction<void> {
    return (dispatch, getState): void => {
        dispatch(hideHotelCrossSaleBlock(tripId));

        localStorage.set(
            HIDDEN_HOTEL_CROSS_SALE_BLOCKS_LS_KEY,
            getState().trips.tripPage.hiddenHotelCrossSaleBlocks,
        );
    };
}

export function restoreHiddenHotelCrossSale(): CustomThunkAction<void> {
    return (dispatch): void => {
        dispatch(
            restoreHiddenHotelsCrossSaleBlocks(
                localStorage.get<string[] | null>(
                    HIDDEN_HOTEL_CROSS_SALE_BLOCKS_LS_KEY,
                ) ?? [],
            ),
        );
    };
}
