import {
    all,
    call,
    cancel,
    delay,
    fork,
    put,
    race,
    select,
    take,
    takeEvery,
} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator, Task} from 'redux-saga';

import {DEFAULT_REFUND_TIMEOUT} from 'projects/account/lib/orders/constants';
import {OPERATION_ORDER_ERROR} from 'constants/errors/orderErrors';

import IGenericOrderGetOrderStateApiResponse from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiResponse';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import * as actions from 'reducers/account/orders/actions';
import {IOrdersState} from 'reducers/account/orders/reducer';

import {trainsOrderTypeSelector} from 'selectors/account/ordersSelector';
import {trainsOrderSelector} from 'selectors/account/order/trains/trainsOrderSelector';

import {
    fetchAndSetRefundAmount,
    refreshRefundAmountAfterUpdateOrder,
} from 'sagas/account/ordersSaga';

import {reachGoal} from 'utilities/metrika';
import genericOrderHasRefundFailed from 'projects/account/lib/orders/genericOrderHasRefundFailed';
import isFinalGenericOrderState from 'projects/trains/lib/api/utilities/isFinalGenericOrderState';
import {unknownErrToString} from 'utilities/error';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

function* orderOrchestratorRefundAmount({
    payload,
}: ReturnType<typeof actions.orderRefundAmountRequest>) {
    const {blankIds, orderId, refundPartContexts} = payload;

    try {
        yield put(actions.setRefundModalOpenedState(true));

        yield fetchAndSetRefundAmount({
            blankIds,
            orderId: orderId,
            refundPartContexts,
        });

        const refreshRefundAmountTask: Task = yield fork(
            refreshRefundAmountAfterUpdateOrder,
            {blankIds, orderId},
        );

        const res: Partial<{
            setRefundModalOpenedState: ReturnType<
                typeof actions.setRefundModalOpenedState
            >;
            startRefund: ReturnType<typeof actions.orderRefund.request>;
        }> = yield race({
            setRefundModalOpenedState: take(
                getType(actions.setRefundModalOpenedState),
            ),
            startRefund: take(getType(actions.orderRefund.request)),
        });

        if (
            res.startRefund ||
            (res.setRefundModalOpenedState &&
                !res.setRefundModalOpenedState.payload)
        ) {
            yield cancel(refreshRefundAmountTask);
        }
    } catch (e) {
        yield put(actions.orderRefund.failure(OPERATION_ORDER_ERROR));
    }
}

function* pollOrderStatusAndSetUpdatedOrder(
    orderId: string,
    initialGenericOrderVersionHash: string | undefined,
) {
    const trainsOrderType: IOrdersState['trainsOrderType'] = yield select(
        trainsOrderTypeSelector,
    );

    if (!trainsOrderType) {
        return;
    }

    try {
        while (true) {
            yield delay(DEFAULT_REFUND_TIMEOUT);

            const genericOrderState: IGenericOrderGetOrderStateApiResponse =
                yield call(genericOrderBrowserProvider.getOrderState, {
                    orderId,
                });

            if (
                initialGenericOrderVersionHash !==
                    genericOrderState.versionHash &&
                isFinalGenericOrderState(genericOrderState.state)
            ) {
                break;
            }
        }

        yield put(
            actions.updateOrder.request({
                orderId,
                trainsOrderType,
                isBackground: true,
            }),
        );

        yield take(getType(actions.updateOrder.success));

        const genericOrder: ReturnType<typeof trainsOrderSelector> =
            yield select(trainsOrderSelector);

        if (genericOrder && genericOrderHasRefundFailed(genericOrder)) {
            yield put(actions.orderRefund.failure(OPERATION_ORDER_ERROR));

            return;
        }

        yield put(actions.setRefundModalOpenedState(false));
    } catch (e) {
        yield put(actions.orderRefund.failure(OPERATION_ORDER_ERROR));
    }
}

function* orderOrchestratorRefund({
    payload,
}: ReturnType<typeof actions.orderRefund.request>) {
    const {orderId, refundToken} = payload;

    try {
        const genericOrderState: IGenericOrderGetOrderStateApiResponse =
            yield call(genericOrderBrowserProvider.getOrderState, {
                orderId,
            });

        yield call(genericOrderBrowserProvider.startRefund, {
            orderId,
            refundToken,
        });

        yield race([
            call(
                pollOrderStatusAndSetUpdatedOrder,
                orderId,
                genericOrderState.versionHash,
            ),
            take(getType(actions.cancelOrderRefundPoll)),
        ]);
    } catch (e) {
        reachGoal(ETrainsGoal.ORDER_REFUND_TICKET_FAILED, {
            trains: {
                refund_error: unknownErrToString(e),
            },
        });

        yield put(actions.orderRefund.failure(OPERATION_ORDER_ERROR));
    }
}

export default function* orderOrchestratorSaga(): SagaIterator {
    yield all([
        yield takeEvery(
            getType(actions.orderRefundAmountRequest),
            orderOrchestratorRefundAmount,
        ),
        yield takeEvery(
            getType(actions.orderRefund.request),
            orderOrchestratorRefund,
        ),
    ]);
}
