import {
    all,
    call,
    delay,
    put,
    select,
    take,
    takeEvery,
} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {
    FETCH_ORDER_ERROR,
    OPERATION_ORDER_ERROR,
} from 'constants/errors/orderErrors';

import {ORDER_TYPE_VALUES} from 'projects/account/lib/orders/types';
import IGenericOrderCalculateRefundAmountApiResponse from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiResponse';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import IRefundPartCtx from 'server/api/GenericOrderApi/types/common/refund/IRefundPartCtx';
import IGenericOrderGetOrderStateApiResponse from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiResponse';
import IGenericOrderGetOrderApiResponse from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiResponse';
import {ORDER_TRAINS_REGISTRATION_STATUS} from 'projects/account/pages/Order/types/OrderTrainsRegistrationStatus';

import * as actions from 'reducers/account/orders/actions';
import {IOrdersState} from 'reducers/account/orders/reducer';

import {trainsOrderTypeSelector} from 'selectors/account/ordersSelector';
import {trainsOrderSelector} from 'selectors/account/order/trains/trainsOrderSelector';

import fetchTrainOrder from 'sagas/account/helpers/fetchTrainOrder';
import getRefundPartContexts from 'sagas/account/helpers/getRefundPartContexts';

import {reachGoal} from 'utilities/metrika';
import isFinalGenericOrderState from 'projects/trains/lib/api/utilities/isFinalGenericOrderState';
import {unknownErrToString} from 'utilities/error';

import {trainsBrowserProvider} from 'serviceProvider/trains/trainsBrowserProvider';
import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

function* fetchAndSetUpdateOrder({
    orderId,
    trainsOrderType,
}: {
    orderId: string;
    trainsOrderType:
        | ORDER_TYPE_VALUES.TRAINS
        | ORDER_TYPE_VALUES.TRAINS_GENERIC;
}) {
    const order: IGenericOrderGetOrderApiResponse = yield call(
        fetchTrainOrder,
        orderId,
        trainsOrderType,
    );

    yield put(actions.updateOrder.success(order));
}

function* orderUpdate({
    payload,
}: ReturnType<typeof actions.updateOrder.request>) {
    const {orderId, trainsOrderType, isBackground} = payload;

    try {
        yield fetchAndSetUpdateOrder({orderId, trainsOrderType});
    } catch (e) {
        if (isBackground) {
            return;
        }

        yield put(actions.updateOrder.failure(FETCH_ORDER_ERROR));
    }
}

interface IFetchAndSetRefundAmountParams {
    orderId: string;

    refundPartContexts?: IRefundPartCtx[];

    /**
     * Для обратной совместимости со старым ЛК
     */
    blankIds?: string[];
}

export function* fetchAndSetRefundAmount(
    params: IFetchAndSetRefundAmountParams,
) {
    const {refundPartContexts} = params;

    const genericOrder: IOrdersState['trainsOrder'] = yield select(
        trainsOrderSelector,
    );

    const refundAmount: IGenericOrderCalculateRefundAmountApiResponse =
        yield call(genericOrderBrowserProvider.calculateRefundAmount, {
            orderId: params.orderId,
            refundPartContexts:
                refundPartContexts ||
                getRefundPartContexts(genericOrder, params.blankIds),
        });

    yield put(actions.setRefundAmountInfo(refundAmount));

    return refundAmount;
}

export function* refreshRefundAmountAfterUpdateOrder(
    params: IFetchAndSetRefundAmountParams,
) {
    yield take(getType(actions.updateOrder.success));

    try {
        yield fetchAndSetRefundAmount(params);
    } catch (e) {
        /**
         * Сильно не расстраиваемся если обновить цену не удалось,
         * продолжаем работать со старой ценой и не ломаем остальные саги
         * https://st.yandex-team.ru/TRAVELFRONT-3045
         */
    }
}

function* orderRegistrationStatusUpdate({
    payload,
}: ReturnType<typeof actions.orderRegistrationStatusUpdate.request>) {
    try {
        const trainsOrderType: IOrdersState['trainsOrderType'] = yield select(
            trainsOrderTypeSelector,
        );

        if (!trainsOrderType) {
            return;
        }

        const {orderId, blankIds, newStatus} = payload;

        const initialOrderState: IGenericOrderGetOrderStateApiResponse =
            yield call(genericOrderBrowserProvider.getOrderState, {
                orderId,
            });

        yield call(trainsBrowserProvider.changeOrderRegistrationStatus, {
            orderId,
            blankIds,
            enabled: newStatus === ORDER_TRAINS_REGISTRATION_STATUS.ENABLED,
        });

        while (true) {
            yield delay(1000);

            const orderState: IGenericOrderGetOrderStateApiResponse =
                yield call(genericOrderBrowserProvider.getOrderState, {
                    orderId,
                });

            if (
                initialOrderState.versionHash !== orderState.versionHash &&
                isFinalGenericOrderState(orderState.state)
            ) {
                break;
            }
        }

        const order: IGenericOrderGetOrderApiResponse = yield call(
            fetchTrainOrder,
            orderId,
            trainsOrderType,
        );

        yield put(actions.orderRegistrationStatusUpdate.success(order));

        reachGoal(ETrainsGoal.ORDER_ELECTRONIC_REGISTRATION_STATUS_CHANGED);
    } catch (e) {
        reachGoal(
            ETrainsGoal.ORDER_ELECTRONIC_REGISTRATION_STATUS_CHANGE_FAILED,
            {
                trains: {
                    change_er_error: unknownErrToString(e),
                },
            },
        );

        yield put(
            actions.orderRegistrationStatusUpdate.failure(
                OPERATION_ORDER_ERROR,
            ),
        );
    }
}

export default function* (): SagaIterator {
    yield all([
        yield takeEvery(getType(actions.updateOrder.request), orderUpdate),
        yield takeEvery(
            getType(actions.orderRegistrationStatusUpdate.request),
            orderRegistrationStatusUpdate,
        ),
    ]);
}
