import {takeEvery, put, all, call} from 'redux-saga/effects';
import {SagaIterator} from 'redux-saga';
import {AxiosError} from 'axios';

import * as docActions from 'reducers/account/docs/actions';
import * as docTypesActions from 'reducers/account/docTypes/actions';
import * as passengerActions from 'reducers/account/passengers/actions';

import {
    fetchDocumentTypes,
    addPassenger,
    editPassenger,
    removePassenger,
} from 'utilities/passengerApiMethods/passengerApiMethods';

import {accountService} from 'serviceProvider';

function* errorHandler(error: AxiosError): SagaIterator | void {
    if (error.response && error.response.status === 401) {
        yield put(passengerActions.authErrorPassenger());
    } else {
        yield put(passengerActions.failPassenger());
    }
}

function* add(
    action: ReturnType<typeof passengerActions.addPassenger>,
): SagaIterator {
    yield put(passengerActions.processPassenger());
    yield put(docActions.processDocument(action.payload.document));

    try {
        const {passenger, document} = yield call(addPassenger, action.payload);

        yield put(passengerActions.addPassengerSuccess(passenger));
        yield put(docActions.addDocumentSuccess(document));
    } catch (e) {
        yield* errorHandler(e);
    }
}

function* addSaga(): SagaIterator {
    yield takeEvery(passengerActions.addPassenger, add);
}

function* edit(
    action: ReturnType<typeof passengerActions.editPassenger>,
): SagaIterator {
    yield put(passengerActions.processPassenger());

    try {
        const result = yield call(editPassenger, action.payload);

        yield put(passengerActions.editPassengerSuccess(result.data));
    } catch (e) {
        yield* errorHandler(e);
    }
}

function* editSaga(): SagaIterator {
    yield takeEvery(passengerActions.editPassenger, edit);
}

function* remove(
    action: ReturnType<typeof passengerActions.removePassenger>,
): SagaIterator {
    yield put(passengerActions.processPassenger());

    try {
        yield call(removePassenger, action.payload.id);
        yield put(
            docActions.removeDocumentSuccess({passengerId: action.payload.id}),
        );
        yield put(passengerActions.removePassengerSuccess({...action.payload}));
    } catch (e) {
        yield* errorHandler(e);
    }
}

function* removeSaga(): SagaIterator {
    yield takeEvery(passengerActions.removePassenger, remove);
}

function* fetch(): SagaIterator | void {
    yield put(passengerActions.processPassenger());

    try {
        const [passengers, {data: documentTypes}] = yield all([
            accountService.provider().getPassengersWithDocumentsAndBonusCards(),
            fetchDocumentTypes(),
        ]);

        yield put(passengerActions.fetchPassengerSuccess(passengers));
        yield put(docTypesActions.docTypesFetchSuccess(documentTypes));
        yield put(docActions.fetchDocumentSuccess(passengers));
    } catch (e) {
        yield* errorHandler(e);
    }
}

function* fetchSaga(): SagaIterator {
    yield takeEvery(passengerActions.fetchPassenger, fetch);
}

export default function* () {
    yield all([addSaga(), editSaga(), removeSaga(), fetchSaga()]);
}
