import {call, put, select, takeLatest} from 'redux-saga/effects';
import _get from 'lodash/get';
import {ActionType, getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {ECalendarType} from 'components/Calendar/types';

import {StoreInterface} from 'reducers/storeTypes';
import {
    calendarPricesFailure,
    fetchCalendarPrices,
    receiveCalendarPrices,
    resetCalendarPrices,
} from 'reducers/avia/aviaCalendarPrices/actions';

import getIsNewCalendarEnabled from 'selectors/common/getIsNewCalendarEnabled';

import getISObyDate from 'components/Calendar/utilities/getISObyDate';

import {aviaPriceIndexBrowserProvider} from 'serviceProvider/avia/aviaPriceIndexBrowserProvider';

const getPriceRequestParams = (state: StoreInterface) => {
    const {from, to, isRoundTrip, startDate, endDate} = state.avia.searchForm;

    return {
        fromId: _get(from, 'selectedValue.pointKey'),
        toId: _get(to, 'selectedValue.pointKey'),
        forwardDate: startDate ? getISObyDate(new Date(startDate)) : undefined,
        backwardDate: endDate ? getISObyDate(new Date(endDate)) : undefined,
        isRoundTrip,
    };
};

const fetchAviaCalendarPrices = function* ({
    payload: calendarType,
}: ActionType<typeof fetchCalendarPrices>): SagaIterator {
    try {
        const {fromId, toId, forwardDate, backwardDate} = yield select(
            getPriceRequestParams,
        );

        const isNewCalendarEnabled = yield select(getIsNewCalendarEnabled);

        if (!toId || !fromId) {
            return yield put(resetCalendarPrices());
        }

        yield put(resetCalendarPrices());

        const newCalendarStartDate =
            isNewCalendarEnabled && !forwardDate && !backwardDate;
        const newCalendarEndDate = isNewCalendarEnabled && forwardDate;

        if (calendarType === ECalendarType.StartDate || newCalendarStartDate) {
            if (backwardDate) {
                const prices = yield call(
                    aviaPriceIndexBrowserProvider.getCalendarPrices,
                    {
                        fromId,
                        toId,
                        forwardDate,
                        backwardDate,
                    },
                );

                yield put(receiveCalendarPrices({prices: prices || {}}));
            } else {
                const prices = yield call(
                    aviaPriceIndexBrowserProvider.getCalendarPrices,
                    {
                        fromId,
                        toId,
                    },
                );

                yield put(receiveCalendarPrices({prices: prices || {}}));
            }
        } else if (
            (calendarType === ECalendarType.EndDate && forwardDate) ||
            newCalendarEndDate
        ) {
            const prices = yield call(
                aviaPriceIndexBrowserProvider.getCalendarPrices,
                {
                    fromId,
                    toId,
                    forwardDate,
                },
            );

            yield put(receiveCalendarPrices({prices: prices || {}}));
        }
    } catch (error) {
        yield put(calendarPricesFailure({error}));
    }
};

export default function* (): SagaIterator {
    yield takeLatest(getType(fetchCalendarPrices), fetchAviaCalendarPrices);
}
