import {take, takeEvery, select, call} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {EAviaActionLogPageName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogPageName';
import {
    IAviaOrderActionLogParamsClient,
    IAviaSearchActionLogParamsClient,
} from 'server/loggers/avia/AviaActionLog/types/IAviaActionLogParamsClient';
import {EAviaActionLogActionName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogActionName';

import {aviaOrderActions} from 'reducers/avia/order/actions';
import {setAviaCurrentPage} from 'reducers/avia/page/actions';
import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {aviaOrderSelectors} from 'selectors/avia/order/aviaOrderSelector';
import {aviaOrderActionLogData} from 'selectors/avia/order/aviaOrderLogSelector';
import {aviaSearchLogSelector} from 'selectors/avia/search/aviaSearchLogSelector';

import {
    logOrderShow,
    logOrderAction,
    logSearchAction,
} from 'projects/avia/lib/logging/aviaVariantsLogger';
import history from 'utilities/browserHistory/browserHistory';
import {unknownToError} from 'utilities/error';
import {locationSearchHasBaggage} from 'projects/avia/lib/urls/queryParams';

import {IAviaOrderRouteProps} from 'projects/avia/pages/AviaOrder/AviaOrder';

import reportError from './reportError';

type TOrderActionLogData = ReturnType<typeof aviaOrderActionLogData>;

interface IAviaOrderRouteExtendedProps extends IAviaOrderRouteProps {
    withBaggage: boolean;
}

function getSelectorProps(): IAviaOrderRouteExtendedProps {
    const location = history?.location;
    const withBaggage = locationSearchHasBaggage(location?.search);

    return {
        variantKey: location?.state?.variantKey,
        withBaggage,
    };
}

function getActionLogParams(
    data: TOrderActionLogData,
): IAviaOrderActionLogParamsClient {
    return {
        page: EAviaActionLogPageName.ORDER,
        qid: data.qid,
        hasOffers: Boolean(data.offers),
        baggage: data.baggage,
        searchIsCompleted: data.searchIsCompleted,
        currency: data.currency,
    };
}

function* logData(
    actionType: EAviaActionLogActionName,
    data: TOrderActionLogData,
): SagaIterator {
    const {offers} = data;
    const variants = offers
        ? [offers.cheapest, ...offers.avia, ...offers.other]
        : [];
    const actionLogParams = getActionLogParams(data);
    const id = yield call(logOrderAction, actionType, actionLogParams);
    const reference = yield select(aviaOrderSelectors.reference);

    yield call(logOrderShow, id, variants, reference);
}

function* baggageWatcher(
    action: ActionType<typeof aviaOrderActions.setBaggageFilter>,
): SagaIterator {
    try {
        if (!action?.meta?.isInitial) {
            const data = yield select(
                aviaOrderActionLogData,
                getSelectorProps(),
            );

            yield call(
                logData,
                EAviaActionLogActionName.ORDER_PAGE_BAGGAGE,
                data,
            );
        }
    } catch (err) {
        reportError(
            unknownToError(err),
            'Ошибка логирования изменения фильтра багажа',
        );
    }
}

function* variantWatcher(
    action: ActionType<typeof aviaOrderActions.updateVariant>,
) {
    try {
        const prevState = action.meta!;
        const prevData: ReturnType<typeof aviaOrderActionLogData> = yield call(
            aviaOrderActionLogData,
            prevState,
            getSelectorProps(),
        );
        const nextData: ReturnType<typeof aviaOrderActionLogData> =
            yield select(aviaOrderActionLogData, getSelectorProps());
        const actionType =
            !prevData.offers && nextData.offers
                ? EAviaActionLogActionName.ORDER_PAGE_LOADED
                : EAviaActionLogActionName.ORDER_PAGE_UPDATE;

        yield logData(actionType, nextData);
    } catch (err) {
        reportError(
            unknownToError(err),
            'Ошибка логирования обновления вариантов',
        );
    }
}

function* pageWatcher({
    payload,
    meta,
}: ActionType<typeof setAviaCurrentPage>): SagaIterator {
    try {
        if (meta) {
            const {
                avia: {
                    page: {entryPage},
                },
            } = meta;

            // Входная страница ещё не была установлена
            if (!entryPage && payload === EAviaActionLogPageName.ORDER) {
                const data = yield select(
                    aviaOrderActionLogData,
                    getSelectorProps(),
                );

                yield call(
                    logData,
                    EAviaActionLogActionName.ORDER_PAGE_SHOW,
                    data,
                );
            }
        }
    } catch (err) {
        reportError(
            unknownToError(err),
            'Ошибка логирования отображения страницы покупки',
        );
    }
}

function* moveToOrderWatcher(): SagaIterator {
    try {
        const searchLogParams: IAviaSearchActionLogParamsClient = yield select(
            aviaSearchLogSelector,
        );
        const id = yield call(
            logSearchAction,
            EAviaActionLogActionName.SEARCH_PAGE_ORDER,
            searchLogParams,
        );

        yield take(setAviaCurrentPage);

        const orderLogParams = yield select(
            aviaOrderActionLogData,
            getSelectorProps(),
        );
        const {offers} = orderLogParams;
        const reference = yield select(aviaOrderSelectors.reference);
        const variants = offers
            ? [offers.cheapest, ...offers.avia, ...offers.other]
            : [];

        yield call(logOrderShow, id, variants, reference);
    } catch (err) {
        reportError(
            unknownToError(err),
            'Ошибка логирования перехода к покупке',
        );
    }
}

export default function* (): SagaIterator {
    yield takeEvery(getType(aviaOrderActions.setBaggageFilter), baggageWatcher);
    yield takeEvery(getType(aviaOrderActions.updateVariant), variantWatcher);
    yield takeEvery(getType(loggerActions.moveToOrder), moveToOrderWatcher);
    yield takeEvery(getType(setAviaCurrentPage), pageWatcher);
}
