import {call, put, select, takeEvery, takeLatest} from 'redux-saga/effects';
import {ActionType, getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {IAviaSearchActionLogParamsClient} from 'server/loggers/avia/AviaActionLog/types/IAviaActionLogParamsClient';
import {EAviaActionLogActionName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogActionName';

import {loggerActions} from 'reducers/avia/aviaLogging/actions';
import {
    initSearch,
    setResultsSorting,
    setTDAnswer,
} from 'reducers/avia/search/results/actions';
import {resetAviaLogMetricsAction} from 'reducers/avia/aviaLogMetrics/actions';

import {aviaSearchLogSelector} from 'selectors/avia/search/aviaSearchLogSelector';
import {getGroupedVariantsInfo} from 'selectors/avia/search/getGroupedVariantsInfo';
import {getAviaSearchReference} from 'selectors/avia/search/aviaSearchResultsSelector';
import {getAviaLogMetrics} from 'selectors/avia/aviaSelectors';
import {aviaPersonalizationSearchHistory} from 'selectors/avia/personalization/aviaPersonalizationSearchHistory';

import {
    logSearchAction,
    logSearchOfferShow,
    logSearchShow,
} from 'projects/avia/lib/logging/aviaVariantsLogger';
import {unknownToError} from 'utilities/error';

import reportError from './reportError';

function* handleShowAction(
    actionId: string,
    showCounter: {count: number},
    {payload}: ActionType<typeof loggerActions.logSearchShow>,
): SagaIterator {
    try {
        if (showCounter.count >= 2) {
            reportError(new Error(), 'Логируются лишние show события');
        }

        const reference = yield select(getAviaSearchReference);

        yield call(
            logSearchShow,
            actionId,
            payload.variants,
            payload.start,
            payload.interesting,
            reference,
        );

        showCounter.count++;
    } catch (err) {
        reportError(unknownToError(err), 'Ошибка записи show лога');
    }
}

function* handleLogAction({
    payload,
}: ActionType<typeof loggerActions.logSearchAction>): SagaIterator {
    try {
        const params: IAviaSearchActionLogParamsClient = yield select(
            aviaSearchLogSelector,
        );
        const id = yield call(logSearchAction, payload.name, {
            ...params,
            ...payload.params,
        });
        const showCounter = {count: 0};

        yield takeEvery(
            getType(loggerActions.logSearchShow),
            handleShowAction.bind(null, id, showCounter),
        );
    } catch (err) {
        reportError(unknownToError(err), 'Ошибка записи action лога');
    }
}

function* handleOfferBaggageFilterAction({
    payload,
}: ActionType<typeof loggerActions.offerBaggageFilter>): SagaIterator {
    try {
        if (payload.variant) {
            const params: IAviaSearchActionLogParamsClient = yield select(
                aviaSearchLogSelector,
            );
            const actionId = yield call(
                logSearchAction,
                EAviaActionLogActionName.SEARCH_PAGE_OFFER_BAGGAGE_FILTER,
                params,
            );
            const reference = yield select(getAviaSearchReference);

            yield call(
                logSearchOfferShow,
                actionId,
                payload.group,
                payload.variant,
                reference,
                payload.position,
                payload.interesting,
            );
        }
    } catch (err) {
        reportError(unknownToError(err), 'Ошибка записи offer baggage события');
    }
}

export default function* (): SagaIterator {
    let variantsWereVisible = false;

    yield takeLatest(getType(loggerActions.logSearchAction), handleLogAction);
    yield takeLatest(
        getType(loggerActions.offerBaggageFilter),
        handleOfferBaggageFilterAction,
    );

    // кастуем logSearchAction а не вызываем напрямую handleLogAction
    // чтобы отключать обработку show событий для предыдущего action события
    // подробней читаем в доке про takeLatest https://redux-saga.js.org/docs/api/
    yield takeEvery(getType(setResultsSorting), function* () {
        yield put(
            loggerActions.logSearchAction({
                name: EAviaActionLogActionName.SEARCH_PAGE_SORT,
            }),
        );
    });
    yield takeEvery(getType(initSearch), function () {
        // при инициализации поиска сбрасываем локальные переменные
        variantsWereVisible = false;
    });
    yield takeEvery(getType(setTDAnswer), function* handleDataSetting() {
        try {
            const variantsInfo: ReturnType<typeof getGroupedVariantsInfo> =
                yield select(getGroupedVariantsInfo);
            const aviaLogMetrics: ReturnType<typeof getAviaLogMetrics> =
                yield select(getAviaLogMetrics);
            const {
                searchHistory,
            }: ReturnType<typeof aviaPersonalizationSearchHistory> =
                yield select(aviaPersonalizationSearchHistory);

            const variantsAreEmpty = variantsInfo.all === 0;
            const noVisibleVariants =
                variantsAreEmpty || variantsInfo.all === variantsInfo.hidden;

            const metrics = variantsWereVisible
                ? undefined
                : {
                      ...aviaLogMetrics,
                      userHasHistory: Boolean(searchHistory.items.length),
                  };

            if (!noVisibleVariants) {
                yield put(
                    loggerActions.logSearchAction({
                        name: variantsWereVisible
                            ? EAviaActionLogActionName.SEARCH_PAGE_UPDATE
                            : EAviaActionLogActionName.SEARCH_PAGE_RESULTS_SHOWN,
                        params: {
                            metrics,
                        },
                    }),
                );

                if (!variantsWereVisible) {
                    yield put(resetAviaLogMetricsAction());
                }
            }

            variantsWereVisible = !noVisibleVariants;
        } catch (err) {
            reportError(
                unknownToError(err),
                'Ошибка обработки события для action лога',
            );
        }
    });
}
