import {call, fork, put, select, takeEvery} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {parse} from 'query-string';
import {SagaIterator} from 'redux-saga';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {
    cleanupCreateOrderState,
    createOrderFailure,
    createOrderRequest,
    createOrderSuccess,
} from 'reducers/avia/booking/createOrder/actions';
import {IPlatformState} from 'reducers/avia/platform/reducer';

import {aviaPlatformSelector} from 'selectors/avia/platform/aviaPlatformSelector';

import history from 'utilities/browserHistory/browserHistory';
import {prepareFormDataToRequest} from 'projects/avia/lib/book/prepareFormDataToRequest';

import {bookingApiClient} from 'projects/avia/pages/AviaBooking/api/bookingApiClient';

import {createCoordinator} from 'contexts/PlatformContext';

import {saveTravelersDocuments} from './saveTravelersDocuments';

function* handleNewOrder({
    payload,
}: ReturnType<typeof createOrderRequest>): SagaIterator {
    yield fork(saveTravelersDocuments, payload);

    try {
        const urlParams = history ? parse(history.location.search) : {};
        const data = prepareFormDataToRequest(payload);

        if (data) {
            const {id} = yield call(bookingApiClient.createOrder, {
                ...data,
                paymentTestContextToken: urlParams.paymentTestContext
                    ? (urlParams.paymentTestContext as string)
                    : undefined,
            });

            const platform: IPlatformState = yield select(aviaPlatformSelector);

            if (platform.isTravelApp) {
                const coordinator = createCoordinator(platform);

                coordinator.doAction(EAppActions.CREATE_AVIA_ORDER, id);

                yield put(cleanupCreateOrderState());

                /**
                 * Полностью отдаем управление Travel App
                 * Сбрасываем состояние страницы до исходных
                 * https://st.yandex-team.ru/TRAVELFRONT-7956
                 */
                return;
            }

            yield put(createOrderSuccess({orderId: id}));
        } else {
            yield put(createOrderFailure(new Error()));
        }
    } catch (ex) {
        yield put(createOrderFailure(ex as Error));
    }
}

export default function* createOrderSaga(): SagaIterator {
    yield takeEvery<ReturnType<typeof createOrderRequest>>(
        getType(createOrderRequest),
        handleNewOrder,
    );
}
