import {put, race, delay, call, take, takeLatest} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {ICompositeOrderStateDTO} from 'server/api/AviaBookingApi/types/ICompositeOrderStateDTO';

import {
    startStatePolling,
    stopStatePolling,
    setCompositeState,
} from 'reducers/avia/booking/compositeStates/actions';
import {orderPaymentPollingFailure} from 'reducers/avia/booking/orderPayment/actions';

import {isErrorOrderState} from 'projects/avia/pages/AviaBooking/lib/orderUtils';

import {bookingApiClient} from 'projects/avia/pages/AviaBooking/api/bookingApiClient';

const POLLING_DELAY = 500;

const MAX_FAILURE_COUNT = 3;

function* orderStatePolling(orderId: string) {
    let failureCount = 0;

    while (true) {
        try {
            const compositeState: ICompositeOrderStateDTO = yield call(
                bookingApiClient.getCompositeState,
                orderId,
            );

            yield put(setCompositeState(orderId, compositeState));

            if (isErrorOrderState(compositeState.orderState)) {
                yield put(stopStatePolling());
            }

            failureCount = 0;
        } catch (ex) {
            failureCount++;

            if (failureCount === MAX_FAILURE_COUNT) {
                yield put(orderPaymentPollingFailure());
                yield put(stopStatePolling());
            }
            /** Ignore errors */
        }

        yield delay(POLLING_DELAY);
    }
}

export default function* (): SagaIterator {
    yield takeLatest<ReturnType<typeof startStatePolling>>(
        getType(startStatePolling),
        function* (action) {
            const {orderId} = action.payload;

            yield race({
                orderState: call(orderStatePolling, orderId),
                cancel: take(getType(stopStatePolling)),
            });
        },
    );
}
