import {call, select} from 'redux-saga/effects';

import {EFieldName} from 'components/BookingPassengerForm/constants/fieldNames';

import {IBookingFormData} from 'projects/avia/pages/AviaBooking/types/IBookingFormData';
import {SagaIterator} from '@redux-saga/types';
import {IBookingFormPassengersData} from 'projects/avia/pages/AviaBooking/types/IBookingFormPassengersData';
import {INotebookSavePassengerParams} from 'server/services/NotebookService/types/INotebookSaveTravelerAndPassengersService';

import {getCode2ToGeoIdConverter} from 'selectors/common/countriesSelectors';

import {parseDate} from 'utilities/dateUtils';
import {HUMAN_DATE_RU, ROBOT} from 'utilities/dateUtils/formats';
import {logError} from 'utilities/logger/logError';
import getTitle from 'utilities/notebook/getTitle';

import {notebookBrowserProvider} from 'serviceProvider/notebook/notebookBrowserProvider';

function getPreparedPassenger(
    passenger: IBookingFormPassengersData,
    code2ToGeoId: ReturnType<typeof getCode2ToGeoIdConverter>,
): INotebookSavePassengerParams {
    const gender = passenger[EFieldName.sex];
    const birthdate = passenger[EFieldName.birthdate];
    const documentType = passenger[EFieldName.documentType];
    const documentNumber = passenger[EFieldName.documentNumber];
    const documentValidDate = passenger[EFieldName.documentValidDate];
    const citizenship = passenger[EFieldName.citizenship];
    const firstNameEn = passenger[EFieldName.firstName];
    const lastNameEn = passenger[EFieldName.lastName];
    const middleNameEn = passenger[EFieldName.patronymicName];
    const geoId = citizenship ? code2ToGeoId(citizenship) : undefined;

    if (
        !gender ||
        !birthdate ||
        !documentType ||
        !geoId ||
        !documentNumber ||
        !firstNameEn ||
        !lastNameEn
    ) {
        throw new Error(
            'У пассажира нет необходимых полей для сохранения в ЗКП',
        );
    }

    return {
        title: getTitle(firstNameEn, lastNameEn, middleNameEn),
        gender,
        birthDate: parseDate(birthdate, HUMAN_DATE_RU).format(ROBOT),
        document: {
            type: documentType,
            citizenship: geoId,
            number: documentNumber,
            expirationDate: documentValidDate
                ? parseDate(documentValidDate, HUMAN_DATE_RU).format(ROBOT)
                : undefined,
            firstNameEn,
            lastNameEn,
            middleNameEn,
        },
        bonusCards: [],
    };
}

export function* saveTravelersDocuments(
    formData: IBookingFormData,
): SagaIterator {
    const code2ToGeoId: ReturnType<typeof getCode2ToGeoIdConverter> =
        yield select(getCode2ToGeoIdConverter);

    const {contacts, passengers} = formData;

    if (!contacts || !passengers) {
        throw new Error('Неверные данные авиа для сохранения в ЗКП');
    }

    try {
        yield call(notebookBrowserProvider.saveTravelerAndPassengers, {
            traveler: {
                email: contacts.email ?? '',
                phone: contacts.phone ?? '',
                agree: true,
            },
            passengers: passengers.map(passenger =>
                getPreparedPassenger(passenger, code2ToGeoId),
            ),
        });
    } catch (err) {
        logError(
            {
                message: '[YATRAVEL][AVIA] Сохранение данных в ЗКП',
            },
            err,
        );
    }
}
