import {ActionType, getType} from 'typesafe-actions';
import {
    call,
    take,
    takeEvery,
    delay,
    put,
    fork,
    cancel,
    select,
} from 'redux-saga/effects';
import {SagaIterator, Task} from 'redux-saga';

import {IAviaTDAnswer} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {TPromiseReturnType} from 'types/utilities';

import {withPrevState} from 'reducers/utils/withPrevState';
import {aviaOrderActions} from 'reducers/avia/order/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {getSearchFormFromContext} from 'projects/avia/lib/search/getSearchFormFromContext';
import {logError} from 'utilities/logger/logError';
import {unknownErrToString, unknownToErrorOrUndefined} from 'utilities/error';

import aviaBrowserProvider from 'serviceProvider/avia/aviaBrowserProvider';

const POLLING_RETRY_COUNT = 10;
const POLLING_DELAY = 2000;
const POLLING_LIMIT = 20000;

function isEmptyTDAnswer(data: IAviaTDAnswer) {
    return !data?.variants?.fares?.length;
}

function* pollOffers({
    payload: qid,
}: ActionType<typeof aviaOrderActions.startUpdateVariant>) {
    if (!qid) {
        yield put(aviaOrderActions.setPartnersDataWereFetched(true));

        return;
    }

    let retriesLeft = POLLING_RETRY_COUNT;
    let cont: number | null = 0;
    let dataWasReceived = false;
    const startTime = Date.now();

    while (cont !== null && Date.now() - startTime < POLLING_LIMIT) {
        try {
            const data: IAviaTDAnswer = yield call(
                aviaBrowserProvider.updateOffers,
                qid,
                cont,
            );

            const tdAnswerIsEmpty = isEmptyTDAnswer(data);

            // Если есть ответ со стороны тикет-демона - берём cont из ответа
            // Если ответ не получили - используем последнее значение
            // TODO: реализовать единый механизм получения цен (https://st.yandex-team.ru/TRAVELFRONT-2999)
            cont = data ? data.cont : cont;
            dataWasReceived = dataWasReceived || !tdAnswerIsEmpty;

            if (!tdAnswerIsEmpty) {
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(withPrevState(aviaOrderActions.updateVariant(data)));
            } else if (data) {
                yield put(aviaOrderActions.updateProgress(data.progress));
            }
        } catch (e) {
            if (retriesLeft-- <= 0) {
                if (dataWasReceived) {
                    cont = null;
                } else {
                    yield put(aviaOrderActions.updateVariantFailure());
                    console.error('update offers error: ', e);

                    return;
                }
            }
        }

        yield delay(POLLING_DELAY);
    }

    yield put(aviaOrderActions.setPartnersDataWereFetched(true));
}

function* pollingManager(
    action: ActionType<typeof aviaOrderActions.startUpdateVariant>,
) {
    const polling: Task = yield fork(pollOffers, action);

    yield take(getType(aviaOrderActions.stopDataFetching));
    yield cancel(polling);
}

function* handleInitSearch() {
    try {
        const reduxState: StoreInterface = yield select();
        const searchForm = getSearchFormFromContext(
            reduxState.avia.aviaContext,
        );

        const result: TPromiseReturnType<
            typeof aviaBrowserProvider.initSearch
        > = yield call(aviaBrowserProvider.initSearch, searchForm);

        yield put(aviaOrderActions.setSearchQid(result.id));

        yield put(aviaOrderActions.startUpdateVariant(result.id));
    } catch (e) {
        logError(
            {
                message: `[YATRAVEL][AVIA] Ошибка инита поиска на покупке: ${unknownErrToString(
                    e,
                )}`,
            },
            unknownToErrorOrUndefined(e),
        );

        yield put(aviaOrderActions.updateVariantFailure());
    }
}

export default function* (): SagaIterator {
    yield takeEvery(
        getType(aviaOrderActions.startUpdateVariant),
        pollingManager,
    );

    yield takeEvery(getType(aviaOrderActions.initSearch), handleInitSearch);
}
